#if HAVE_CRT
#define _CRTDBG_MAP_ALLOC 
#include <stdlib.h> 
#include <crtdbg.h>
#endif //HAVE_CRT

/* #line 1 "./ragel/tsip_parser_header_Referred_By.rl" */

/*
* Copyright (C) 2020, University of the Basque Country (UPV/EHU)
* Contact for licensing options: <licensing-mcpttclient(at)mcopenplatform(dot)com>
*
* The original file was part of Open Source Doubango Framework
* Copyright (C) 2010-2011 Mamadou Diop.
* Copyright (C) 2012 Doubango Telecom <http://doubango.org>
*
* This file is part of Open Source Doubango Framework.
*
* DOUBANGO is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* DOUBANGO is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with DOUBANGO.
*
*/


/**@file tsip_header_Referred_By.c
 * @brief SIP Referred-By header as per RFC 3892.
 *
 * @author Mamadou Diop <diopmamadou(at)doubango[dot]org>
 *

 */
#include "tinysip/headers/tsip_header_Referred_By.h"

#include "tinysip/parsers/tsip_parser_uri.h"

#include "tsk_debug.h"
#include "tsk_memory.h"



/***********************************
*	Ragel state machine.
*/

/* #line 95 "./ragel/tsip_parser_header_Referred_By.rl" */



tsip_header_Referred_By_t* tsip_header_Referred_By_create(const tsip_uri_t* uri, const char* cid)
{
	return tsk_object_new(TSIP_HEADER_REFERRED_BY_VA_ARGS(uri, cid));
}

tsip_header_Referred_By_t* tsip_header_Referred_By_create_null()
{
	return tsip_header_Referred_By_create(tsk_null, tsk_null);
}

int tsip_header_Referred_By_serialize(const tsip_header_t* header, tsk_buffer_t* output)
{
	if(header){
		int ret;
		const tsip_header_Referred_By_t *Referred_By = (const tsip_header_Referred_By_t *)header;

		/* Uri with hacked display-name*/
		if((ret = tsip_uri_serialize(Referred_By->uri, tsk_true, tsk_true, output))){
			return ret;
		}
		/* cid */
		if(Referred_By->cid && (ret = tsk_buffer_append_2(output, ";cid=%s", Referred_By->cid))){
			return ret;
		}
		return ret;
	}
	return -1;
}

tsip_header_Referred_By_t *tsip_header_Referred_By_parse(const char *data, tsk_size_t size)
{
	int cs = 0;
	const char *p = data;
	const char *pe = p + size;
	const char *eof = pe;
	tsip_header_Referred_By_t *r_by = tsip_header_Referred_By_create_null();
	
	const char *tag_start = tsk_null;

	
/* #line 91 "./src/headers/tsip_header_Referred_By.c" */
static const char _tsip_machine_parser_header_Referred_By_actions[] = {
	0, 1, 0, 1, 1, 1, 2, 1, 
	3, 1, 4, 1, 5, 2, 1, 0, 
	2, 1, 3, 2, 1, 4, 2, 3, 
	4, 3, 1, 3, 4
};

static const short _tsip_machine_parser_header_Referred_By_key_offsets[] = {
	0, 0, 4, 7, 28, 29, 31, 52, 
	53, 55, 58, 62, 74, 77, 77, 78, 
	82, 83, 102, 103, 105, 123, 142, 147, 
	148, 150, 154, 173, 174, 176, 195, 196, 
	198, 201, 209, 210, 212, 216, 220, 221, 
	223, 226, 232, 250, 257, 265, 273, 281, 
	283, 290, 299, 301, 304, 306, 309, 311, 
	314, 317, 318, 321, 322, 325, 326, 335, 
	344, 352, 360, 368, 376, 378, 384, 393, 
	402, 411, 413, 416, 419, 420, 421, 442, 
	463, 482, 487, 488, 490, 494, 513, 514, 
	516, 535, 536, 538, 541, 562, 585, 609, 
	631, 635, 639, 660, 684, 708, 731, 755, 
	779, 802, 817, 833, 849, 865, 875, 890, 
	907, 917, 928, 938, 949, 959, 970, 981, 
	990, 998, 1009, 1018, 1029, 1038, 1055, 1072, 
	1088, 1104, 1120, 1136, 1146, 1160, 1177, 1194, 
	1211, 1221, 1232, 1243, 1252, 1261, 1278, 1296, 
	1300, 1301, 1303, 1311, 1312, 1314, 1318, 1324, 
	1336, 1339, 1339, 1343, 1363, 1382, 1387, 1407, 
	1425, 1434, 1438, 1463, 1487, 1497, 1521, 1544, 
	1552, 1561, 1577, 1594, 1611, 1628, 1639, 1655, 
	1673, 1684, 1696, 1707, 1719, 1730, 1742, 1754, 
	1764, 1776, 1786, 1798, 1808, 1826, 1844, 1861, 
	1878, 1895, 1912, 1923, 1938, 1956, 1974, 1992, 
	2003, 2015, 2027, 2037, 2047, 2073, 2099, 2123, 
	2133, 2157, 2179, 2203, 2225, 2250, 2273, 2277, 
	2281, 2303, 2328, 2353, 2377, 2402, 2427, 2451, 
	2467, 2484, 2501, 2518, 2529, 2545, 2563, 2574, 
	2586, 2597, 2609, 2620, 2632, 2644, 2654, 2663, 
	2675, 2685, 2697, 2707, 2725, 2743, 2760, 2777, 
	2794, 2811, 2822, 2837, 2855, 2873, 2891, 2902, 
	2914, 2926, 2936, 2946, 2957, 2969, 2981, 2993, 
	2999, 3010, 3023, 3029, 3036, 3042, 3049, 3055, 
	3062, 3069, 3074, 3081, 3086, 3093, 3098, 3111, 
	3124, 3136, 3148, 3160, 3172, 3178, 3188, 3201, 
	3214, 3227, 3233, 3240, 3247, 3252, 3257, 3278, 
	3299, 3318, 3323, 3343, 3345, 3347, 3349, 3351, 
	3353, 3355, 3357, 3358, 3360, 3362
};

static const char _tsip_machine_parser_header_Referred_By_trans_keys[] = {
	66, 82, 98, 114, 9, 32, 58, 9, 
	13, 32, 33, 34, 37, 39, 60, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 96, 97, 122, 10, 9, 32, 9, 
	13, 32, 33, 34, 37, 39, 60, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 96, 97, 122, 10, 9, 32, 9, 
	32, 60, 65, 90, 97, 122, 9, 32, 
	43, 58, 45, 46, 48, 57, 65, 90, 
	97, 122, 9, 32, 58, 62, 9, 13, 
	32, 59, 10, 9, 13, 32, 33, 37, 
	39, 67, 99, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 10, 9, 
	32, 9, 32, 33, 37, 39, 67, 99, 
	126, 42, 43, 45, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 59, 61, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 59, 61, 10, 9, 32, 9, 32, 
	59, 61, 9, 13, 32, 33, 34, 37, 
	39, 91, 126, 42, 43, 45, 46, 48, 
	57, 65, 90, 95, 122, 10, 9, 32, 
	9, 13, 32, 33, 34, 37, 39, 91, 
	126, 42, 43, 45, 46, 48, 57, 65, 
	90, 95, 122, 10, 9, 32, 9, 32, 
	34, 13, 34, 92, 127, 0, 8, 10, 
	31, 10, 9, 32, 9, 13, 32, 59, 
	9, 13, 32, 59, 10, 9, 32, 9, 
	32, 59, 0, 9, 11, 12, 14, 127, 
	9, 13, 32, 33, 37, 39, 59, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 122, 58, 48, 57, 65, 70, 97, 
	102, 58, 93, 48, 57, 65, 70, 97, 
	102, 58, 93, 48, 57, 65, 70, 97, 
	102, 58, 93, 48, 57, 65, 70, 97, 
	102, 58, 93, 58, 48, 57, 65, 70, 
	97, 102, 46, 58, 93, 48, 57, 65, 
	70, 97, 102, 48, 57, 46, 48, 57, 
	48, 57, 46, 48, 57, 48, 57, 93, 
	48, 57, 93, 48, 57, 93, 46, 48, 
	57, 46, 46, 48, 57, 46, 46, 58, 
	93, 48, 57, 65, 70, 97, 102, 46, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	46, 58, 93, 48, 57, 65, 70, 97, 
	102, 46, 58, 93, 48, 57, 65, 70, 
	97, 102, 46, 58, 93, 48, 57, 65, 
	70, 97, 102, 48, 57, 46, 48, 57, 
	46, 48, 57, 46, 58, 9, 13, 32, 
	33, 37, 39, 59, 61, 73, 105, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 122, 9, 13, 32, 33, 37, 39, 
	59, 61, 68, 100, 126, 42, 43, 45, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 59, 61, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 122, 9, 13, 32, 59, 61, 10, 
	9, 32, 9, 32, 59, 61, 9, 13, 
	32, 33, 34, 37, 39, 91, 126, 42, 
	43, 45, 46, 48, 57, 65, 90, 95, 
	122, 10, 9, 32, 9, 13, 32, 33, 
	34, 37, 39, 91, 126, 42, 43, 45, 
	46, 48, 57, 65, 90, 95, 122, 10, 
	9, 32, 9, 32, 34, 13, 33, 34, 
	37, 39, 45, 92, 126, 127, 0, 8, 
	10, 31, 42, 43, 48, 57, 65, 90, 
	95, 122, 13, 33, 34, 37, 39, 45, 
	46, 64, 92, 126, 127, 0, 8, 10, 
	31, 42, 43, 48, 57, 65, 90, 95, 
	122, 13, 33, 34, 37, 39, 45, 91, 
	92, 126, 127, 0, 8, 10, 31, 42, 
	43, 48, 57, 65, 90, 95, 96, 97, 
	122, 13, 33, 34, 37, 39, 45, 46, 
	92, 126, 127, 0, 8, 10, 31, 42, 
	43, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 59, 9, 13, 32, 59, 13, 
	33, 34, 37, 39, 45, 92, 126, 127, 
	0, 8, 10, 31, 42, 43, 48, 57, 
	65, 90, 95, 122, 13, 33, 34, 37, 
	39, 45, 46, 92, 126, 127, 0, 8, 
	10, 31, 42, 43, 48, 57, 65, 90, 
	95, 96, 97, 122, 13, 33, 34, 37, 
	39, 45, 46, 92, 126, 127, 0, 8, 
	10, 31, 42, 43, 48, 57, 65, 90, 
	95, 96, 97, 122, 13, 33, 34, 37, 
	39, 45, 92, 126, 127, 0, 8, 10, 
	31, 42, 43, 48, 57, 65, 90, 95, 
	96, 97, 122, 13, 33, 34, 37, 39, 
	45, 46, 92, 126, 127, 0, 8, 10, 
	31, 42, 43, 48, 57, 65, 90, 95, 
	96, 97, 122, 13, 33, 34, 37, 39, 
	45, 46, 92, 126, 127, 0, 8, 10, 
	31, 42, 43, 48, 57, 65, 90, 95, 
	96, 97, 122, 13, 33, 34, 37, 39, 
	45, 92, 126, 127, 0, 8, 10, 31, 
	42, 43, 48, 57, 65, 90, 95, 96, 
	97, 122, 13, 34, 58, 92, 127, 0, 
	8, 10, 31, 48, 57, 65, 70, 97, 
	102, 13, 34, 58, 92, 93, 127, 0, 
	8, 10, 31, 48, 57, 65, 70, 97, 
	102, 13, 34, 58, 92, 93, 127, 0, 
	8, 10, 31, 48, 57, 65, 70, 97, 
	102, 13, 34, 58, 92, 93, 127, 0, 
	8, 10, 31, 48, 57, 65, 70, 97, 
	102, 13, 34, 58, 92, 93, 127, 0, 
	8, 10, 31, 13, 34, 58, 92, 127, 
	0, 8, 10, 31, 48, 57, 65, 70, 
	97, 102, 13, 34, 46, 58, 92, 93, 
	127, 0, 8, 10, 31, 48, 57, 65, 
	70, 97, 102, 13, 34, 92, 127, 0, 
	8, 10, 31, 48, 57, 13, 34, 46, 
	92, 127, 0, 8, 10, 31, 48, 57, 
	13, 34, 92, 127, 0, 8, 10, 31, 
	48, 57, 13, 34, 46, 92, 127, 0, 
	8, 10, 31, 48, 57, 13, 34, 92, 
	127, 0, 8, 10, 31, 48, 57, 13, 
	34, 92, 93, 127, 0, 8, 10, 31, 
	48, 57, 13, 34, 92, 93, 127, 0, 
	8, 10, 31, 48, 57, 13, 34, 92, 
	93, 127, 0, 8, 10, 31, 13, 34, 
	92, 127, 0, 8, 10, 31, 13, 34, 
	46, 92, 127, 0, 8, 10, 31, 48, 
	57, 13, 34, 46, 92, 127, 0, 8, 
	10, 31, 13, 34, 46, 92, 127, 0, 
	8, 10, 31, 48, 57, 13, 34, 46, 
	92, 127, 0, 8, 10, 31, 13, 34, 
	46, 58, 92, 93, 127, 0, 8, 10, 
	31, 48, 57, 65, 70, 97, 102, 13, 
	34, 46, 58, 92, 93, 127, 0, 8, 
	10, 31, 48, 57, 65, 70, 97, 102, 
	13, 34, 58, 92, 93, 127, 0, 8, 
	10, 31, 48, 57, 65, 70, 97, 102, 
	13, 34, 58, 92, 93, 127, 0, 8, 
	10, 31, 48, 57, 65, 70, 97, 102, 
	13, 34, 58, 92, 93, 127, 0, 8, 
	10, 31, 48, 57, 65, 70, 97, 102, 
	13, 34, 58, 92, 93, 127, 0, 8, 
	10, 31, 48, 57, 65, 70, 97, 102, 
	13, 34, 58, 92, 93, 127, 0, 8, 
	10, 31, 13, 34, 92, 127, 0, 8, 
	10, 31, 48, 57, 65, 70, 97, 102, 
	13, 34, 46, 58, 92, 93, 127, 0, 
	8, 10, 31, 48, 57, 65, 70, 97, 
	102, 13, 34, 46, 58, 92, 93, 127, 
	0, 8, 10, 31, 48, 57, 65, 70, 
	97, 102, 13, 34, 46, 58, 92, 93, 
	127, 0, 8, 10, 31, 48, 57, 65, 
	70, 97, 102, 13, 34, 92, 127, 0, 
	8, 10, 31, 48, 57, 13, 34, 46, 
	92, 127, 0, 8, 10, 31, 48, 57, 
	13, 34, 46, 92, 127, 0, 8, 10, 
	31, 48, 57, 13, 34, 46, 92, 127, 
	0, 8, 10, 31, 13, 34, 58, 92, 
	127, 0, 8, 10, 31, 9, 13, 32, 
	33, 37, 39, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 33, 37, 39, 60, 126, 42, 43, 
	45, 46, 48, 57, 65, 90, 95, 122, 
	9, 13, 32, 60, 10, 9, 32, 13, 
	34, 92, 127, 0, 8, 10, 31, 10, 
	9, 32, 9, 13, 32, 60, 0, 9, 
	11, 12, 14, 127, 9, 32, 43, 58, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	9, 32, 58, 9, 13, 32, 59, 9, 
	13, 32, 33, 37, 39, 59, 67, 99, 
	126, 42, 43, 45, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 59, 61, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 59, 61, 9, 13, 32, 33, 34, 
	37, 39, 59, 91, 126, 42, 43, 45, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 59, 126, 42, 
	43, 45, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 34, 59, 92, 127, 
	0, 31, 9, 13, 32, 59, 9, 13, 
	32, 33, 34, 37, 39, 59, 67, 92, 
	99, 126, 127, 0, 31, 42, 43, 45, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 34, 37, 39, 59, 61, 
	92, 126, 127, 0, 31, 42, 43, 45, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 34, 59, 61, 92, 127, 0, 
	31, 9, 13, 32, 33, 34, 37, 39, 
	59, 91, 92, 126, 127, 0, 31, 42, 
	43, 45, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 33, 34, 37, 39, 
	59, 92, 126, 127, 0, 31, 42, 43, 
	45, 46, 48, 57, 65, 90, 95, 122, 
	9, 13, 32, 59, 0, 8, 11, 127, 
	9, 13, 32, 34, 59, 92, 127, 0, 
	31, 9, 13, 32, 34, 58, 59, 92, 
	127, 0, 31, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 34, 58, 59, 92, 
	93, 127, 0, 31, 48, 57, 65, 70, 
	97, 102, 9, 13, 32, 34, 58, 59, 
	92, 93, 127, 0, 31, 48, 57, 65, 
	70, 97, 102, 9, 13, 32, 34, 58, 
	59, 92, 93, 127, 0, 31, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 34, 
	58, 59, 92, 93, 127, 0, 31, 9, 
	13, 32, 34, 58, 59, 92, 127, 0, 
	31, 48, 57, 65, 70, 97, 102, 9, 
	13, 32, 34, 46, 58, 59, 92, 93, 
	127, 0, 31, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 34, 59, 92, 127, 
	0, 31, 48, 57, 9, 13, 32, 34, 
	46, 59, 92, 127, 0, 31, 48, 57, 
	9, 13, 32, 34, 59, 92, 127, 0, 
	31, 48, 57, 9, 13, 32, 34, 46, 
	59, 92, 127, 0, 31, 48, 57, 9, 
	13, 32, 34, 59, 92, 127, 0, 31, 
	48, 57, 9, 13, 32, 34, 59, 92, 
	93, 127, 0, 31, 48, 57, 9, 13, 
	32, 34, 59, 92, 93, 127, 0, 31, 
	48, 57, 9, 13, 32, 34, 59, 92, 
	93, 127, 0, 31, 9, 13, 32, 34, 
	46, 59, 92, 127, 0, 31, 48, 57, 
	9, 13, 32, 34, 46, 59, 92, 127, 
	0, 31, 9, 13, 32, 34, 46, 59, 
	92, 127, 0, 31, 48, 57, 9, 13, 
	32, 34, 46, 59, 92, 127, 0, 31, 
	9, 13, 32, 34, 46, 58, 59, 92, 
	93, 127, 0, 31, 48, 57, 65, 70, 
	97, 102, 9, 13, 32, 34, 46, 58, 
	59, 92, 93, 127, 0, 31, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 34, 
	58, 59, 92, 93, 127, 0, 31, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	34, 58, 59, 92, 93, 127, 0, 31, 
	48, 57, 65, 70, 97, 102, 9, 13, 
	32, 34, 58, 59, 92, 93, 127, 0, 
	31, 48, 57, 65, 70, 97, 102, 9, 
	13, 32, 34, 58, 59, 92, 93, 127, 
	0, 31, 48, 57, 65, 70, 97, 102, 
	9, 13, 32, 34, 58, 59, 92, 93, 
	127, 0, 31, 9, 13, 32, 34, 59, 
	92, 127, 0, 31, 48, 57, 65, 70, 
	97, 102, 9, 13, 32, 34, 46, 58, 
	59, 92, 93, 127, 0, 31, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 34, 
	46, 58, 59, 92, 93, 127, 0, 31, 
	48, 57, 65, 70, 97, 102, 9, 13, 
	32, 34, 46, 58, 59, 92, 93, 127, 
	0, 31, 48, 57, 65, 70, 97, 102, 
	9, 13, 32, 34, 59, 92, 127, 0, 
	31, 48, 57, 9, 13, 32, 34, 46, 
	59, 92, 127, 0, 31, 48, 57, 9, 
	13, 32, 34, 46, 59, 92, 127, 0, 
	31, 48, 57, 9, 13, 32, 34, 46, 
	59, 92, 127, 0, 31, 9, 13, 32, 
	34, 58, 59, 92, 127, 0, 31, 9, 
	13, 32, 33, 34, 37, 39, 59, 61, 
	73, 92, 105, 126, 127, 0, 31, 42, 
	43, 45, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 33, 34, 37, 39, 
	59, 61, 68, 92, 100, 126, 127, 0, 
	31, 42, 43, 45, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 34, 
	37, 39, 59, 61, 92, 126, 127, 0, 
	31, 42, 43, 45, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 34, 59, 
	61, 92, 127, 0, 31, 9, 13, 32, 
	33, 34, 37, 39, 59, 91, 92, 126, 
	127, 0, 31, 42, 43, 45, 46, 48, 
	57, 65, 90, 95, 122, 9, 13, 32, 
	33, 34, 37, 39, 45, 59, 92, 126, 
	127, 0, 31, 42, 43, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 34, 
	37, 39, 45, 46, 59, 64, 92, 126, 
	127, 0, 31, 42, 43, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 34, 
	37, 39, 45, 59, 92, 126, 127, 0, 
	31, 42, 43, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 33, 34, 37, 39, 
	45, 59, 91, 92, 126, 127, 0, 31, 
	42, 43, 48, 57, 65, 90, 95, 96, 
	97, 122, 9, 13, 32, 33, 34, 37, 
	39, 45, 46, 59, 92, 126, 127, 0, 
	31, 42, 43, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 59, 9, 13, 32, 
	59, 9, 13, 32, 33, 34, 37, 39, 
	45, 59, 92, 126, 127, 0, 31, 42, 
	43, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 34, 37, 39, 45, 46, 
	59, 92, 126, 127, 0, 31, 42, 43, 
	48, 57, 65, 90, 95, 96, 97, 122, 
	9, 13, 32, 33, 34, 37, 39, 45, 
	46, 59, 92, 126, 127, 0, 31, 42, 
	43, 48, 57, 65, 90, 95, 96, 97, 
	122, 9, 13, 32, 33, 34, 37, 39, 
	45, 59, 92, 126, 127, 0, 31, 42, 
	43, 48, 57, 65, 90, 95, 96, 97, 
	122, 9, 13, 32, 33, 34, 37, 39, 
	45, 46, 59, 92, 126, 127, 0, 31, 
	42, 43, 48, 57, 65, 90, 95, 96, 
	97, 122, 9, 13, 32, 33, 34, 37, 
	39, 45, 46, 59, 92, 126, 127, 0, 
	31, 42, 43, 48, 57, 65, 90, 95, 
	96, 97, 122, 9, 13, 32, 33, 34, 
	37, 39, 45, 59, 92, 126, 127, 0, 
	31, 42, 43, 48, 57, 65, 90, 95, 
	96, 97, 122, 9, 13, 32, 34, 58, 
	59, 92, 127, 0, 31, 48, 57, 65, 
	70, 97, 102, 9, 13, 32, 34, 58, 
	59, 92, 93, 127, 0, 31, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 34, 
	58, 59, 92, 93, 127, 0, 31, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	34, 58, 59, 92, 93, 127, 0, 31, 
	48, 57, 65, 70, 97, 102, 9, 13, 
	32, 34, 58, 59, 92, 93, 127, 0, 
	31, 9, 13, 32, 34, 58, 59, 92, 
	127, 0, 31, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 34, 46, 58, 59, 
	92, 93, 127, 0, 31, 48, 57, 65, 
	70, 97, 102, 9, 13, 32, 34, 59, 
	92, 127, 0, 31, 48, 57, 9, 13, 
	32, 34, 46, 59, 92, 127, 0, 31, 
	48, 57, 9, 13, 32, 34, 59, 92, 
	127, 0, 31, 48, 57, 9, 13, 32, 
	34, 46, 59, 92, 127, 0, 31, 48, 
	57, 9, 13, 32, 34, 59, 92, 127, 
	0, 31, 48, 57, 9, 13, 32, 34, 
	59, 92, 93, 127, 0, 31, 48, 57, 
	9, 13, 32, 34, 59, 92, 93, 127, 
	0, 31, 48, 57, 9, 13, 32, 34, 
	59, 92, 93, 127, 0, 31, 9, 13, 
	32, 34, 59, 92, 127, 0, 31, 9, 
	13, 32, 34, 46, 59, 92, 127, 0, 
	31, 48, 57, 9, 13, 32, 34, 46, 
	59, 92, 127, 0, 31, 9, 13, 32, 
	34, 46, 59, 92, 127, 0, 31, 48, 
	57, 9, 13, 32, 34, 46, 59, 92, 
	127, 0, 31, 9, 13, 32, 34, 46, 
	58, 59, 92, 93, 127, 0, 31, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	34, 46, 58, 59, 92, 93, 127, 0, 
	31, 48, 57, 65, 70, 97, 102, 9, 
	13, 32, 34, 58, 59, 92, 93, 127, 
	0, 31, 48, 57, 65, 70, 97, 102, 
	9, 13, 32, 34, 58, 59, 92, 93, 
	127, 0, 31, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 34, 58, 59, 92, 
	93, 127, 0, 31, 48, 57, 65, 70, 
	97, 102, 9, 13, 32, 34, 58, 59, 
	92, 93, 127, 0, 31, 48, 57, 65, 
	70, 97, 102, 9, 13, 32, 34, 58, 
	59, 92, 93, 127, 0, 31, 9, 13, 
	32, 34, 59, 92, 127, 0, 31, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	34, 46, 58, 59, 92, 93, 127, 0, 
	31, 48, 57, 65, 70, 97, 102, 9, 
	13, 32, 34, 46, 58, 59, 92, 93, 
	127, 0, 31, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 34, 46, 58, 59, 
	92, 93, 127, 0, 31, 48, 57, 65, 
	70, 97, 102, 9, 13, 32, 34, 59, 
	92, 127, 0, 31, 48, 57, 9, 13, 
	32, 34, 46, 59, 92, 127, 0, 31, 
	48, 57, 9, 13, 32, 34, 46, 59, 
	92, 127, 0, 31, 48, 57, 9, 13, 
	32, 34, 46, 59, 92, 127, 0, 31, 
	9, 13, 32, 34, 58, 59, 92, 127, 
	0, 31, 9, 13, 32, 58, 59, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	58, 59, 93, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 58, 59, 93, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	58, 59, 93, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 58, 59, 93, 9, 
	13, 32, 58, 59, 48, 57, 65, 70, 
	97, 102, 9, 13, 32, 46, 58, 59, 
	93, 48, 57, 65, 70, 97, 102, 9, 
	13, 32, 59, 48, 57, 9, 13, 32, 
	46, 59, 48, 57, 9, 13, 32, 59, 
	48, 57, 9, 13, 32, 46, 59, 48, 
	57, 9, 13, 32, 59, 48, 57, 9, 
	13, 32, 59, 93, 48, 57, 9, 13, 
	32, 59, 93, 48, 57, 9, 13, 32, 
	59, 93, 9, 13, 32, 46, 59, 48, 
	57, 9, 13, 32, 46, 59, 9, 13, 
	32, 46, 59, 48, 57, 9, 13, 32, 
	46, 59, 9, 13, 32, 46, 58, 59, 
	93, 48, 57, 65, 70, 97, 102, 9, 
	13, 32, 46, 58, 59, 93, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 58, 
	59, 93, 48, 57, 65, 70, 97, 102, 
	9, 13, 32, 58, 59, 93, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 58, 
	59, 93, 48, 57, 65, 70, 97, 102, 
	9, 13, 32, 58, 59, 93, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 58, 
	59, 93, 9, 13, 32, 59, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 46, 
	58, 59, 93, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 46, 58, 59, 93, 
	48, 57, 65, 70, 97, 102, 9, 13, 
	32, 46, 58, 59, 93, 48, 57, 65, 
	70, 97, 102, 9, 13, 32, 59, 48, 
	57, 9, 13, 32, 46, 59, 48, 57, 
	9, 13, 32, 46, 59, 48, 57, 9, 
	13, 32, 46, 59, 9, 13, 32, 58, 
	59, 9, 13, 32, 33, 37, 39, 59, 
	61, 73, 105, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 33, 37, 39, 59, 61, 68, 100, 
	126, 42, 43, 45, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 59, 61, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 59, 61, 9, 13, 32, 33, 34, 
	37, 39, 59, 91, 126, 42, 43, 45, 
	46, 48, 57, 65, 90, 95, 122, 69, 
	101, 70, 102, 69, 101, 82, 114, 82, 
	114, 69, 101, 68, 100, 45, 66, 98, 
	89, 121, 0
};

static const char _tsip_machine_parser_header_Referred_By_single_lengths[] = {
	0, 4, 3, 9, 1, 2, 9, 1, 
	2, 3, 0, 4, 3, 0, 1, 4, 
	1, 9, 1, 2, 8, 9, 5, 1, 
	2, 4, 9, 1, 2, 9, 1, 2, 
	3, 4, 1, 2, 4, 4, 1, 2, 
	3, 0, 8, 1, 2, 2, 2, 2, 
	1, 3, 0, 1, 0, 1, 0, 1, 
	1, 1, 1, 1, 1, 1, 3, 3, 
	2, 2, 2, 2, 2, 0, 3, 3, 
	3, 0, 1, 1, 1, 1, 11, 11, 
	9, 5, 1, 2, 4, 9, 1, 2, 
	9, 1, 2, 3, 9, 11, 10, 10, 
	4, 4, 9, 10, 10, 9, 10, 10, 
	9, 5, 6, 6, 6, 6, 5, 7, 
	4, 5, 4, 5, 4, 5, 5, 5, 
	4, 5, 5, 5, 5, 7, 7, 6, 
	6, 6, 6, 6, 4, 7, 7, 7, 
	4, 5, 5, 5, 5, 7, 8, 4, 
	1, 2, 4, 1, 2, 4, 0, 4, 
	3, 0, 4, 10, 9, 5, 10, 8, 
	7, 4, 13, 12, 8, 12, 11, 4, 
	7, 8, 9, 9, 9, 9, 8, 10, 
	7, 8, 7, 8, 7, 8, 8, 8, 
	8, 8, 8, 8, 10, 10, 9, 9, 
	9, 9, 9, 7, 10, 10, 10, 7, 
	8, 8, 8, 8, 14, 14, 12, 8, 
	12, 12, 14, 12, 13, 13, 4, 4, 
	12, 13, 13, 12, 13, 13, 12, 8, 
	9, 9, 9, 9, 8, 10, 7, 8, 
	7, 8, 7, 8, 8, 8, 7, 8, 
	8, 8, 8, 10, 10, 9, 9, 9, 
	9, 9, 7, 10, 10, 10, 7, 8, 
	8, 8, 8, 5, 6, 6, 6, 6, 
	5, 7, 4, 5, 4, 5, 4, 5, 
	5, 5, 5, 5, 5, 5, 7, 7, 
	6, 6, 6, 6, 6, 4, 7, 7, 
	7, 4, 5, 5, 5, 5, 11, 11, 
	9, 5, 10, 2, 2, 2, 2, 2, 
	2, 2, 1, 2, 2, 0
};

static const char _tsip_machine_parser_header_Referred_By_range_lengths[] = {
	0, 0, 0, 6, 0, 0, 6, 0, 
	0, 0, 2, 4, 0, 0, 0, 0, 
	0, 5, 0, 0, 5, 5, 0, 0, 
	0, 0, 5, 0, 0, 5, 0, 0, 
	0, 2, 0, 0, 0, 0, 0, 0, 
	0, 3, 5, 3, 3, 3, 3, 0, 
	3, 3, 1, 1, 1, 1, 1, 1, 
	1, 0, 1, 0, 1, 0, 3, 3, 
	3, 3, 3, 3, 0, 3, 3, 3, 
	3, 1, 1, 1, 0, 0, 5, 5, 
	5, 0, 0, 0, 0, 5, 0, 0, 
	5, 0, 0, 0, 6, 6, 7, 6, 
	0, 0, 6, 7, 7, 7, 7, 7, 
	7, 5, 5, 5, 5, 2, 5, 5, 
	3, 3, 3, 3, 3, 3, 3, 2, 
	2, 3, 2, 3, 2, 5, 5, 5, 
	5, 5, 5, 2, 5, 5, 5, 5, 
	3, 3, 3, 2, 2, 5, 5, 0, 
	0, 0, 2, 0, 0, 0, 3, 4, 
	0, 0, 0, 5, 5, 0, 5, 5, 
	1, 0, 6, 6, 1, 6, 6, 2, 
	1, 4, 4, 4, 4, 1, 4, 4, 
	2, 2, 2, 2, 2, 2, 2, 1, 
	2, 1, 2, 1, 4, 4, 4, 4, 
	4, 4, 1, 4, 4, 4, 4, 2, 
	2, 2, 1, 1, 6, 6, 6, 1, 
	6, 5, 5, 5, 6, 5, 0, 0, 
	5, 6, 6, 6, 6, 6, 6, 4, 
	4, 4, 4, 1, 4, 4, 2, 2, 
	2, 2, 2, 2, 2, 1, 1, 2, 
	1, 2, 1, 4, 4, 4, 4, 4, 
	4, 1, 4, 4, 4, 4, 2, 2, 
	2, 1, 1, 3, 3, 3, 3, 0, 
	3, 3, 1, 1, 1, 1, 1, 1, 
	1, 0, 1, 0, 1, 0, 3, 3, 
	3, 3, 3, 3, 0, 3, 3, 3, 
	3, 1, 1, 1, 0, 0, 5, 5, 
	5, 0, 5, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0
};

static const short _tsip_machine_parser_header_Referred_By_index_offsets[] = {
	0, 0, 5, 9, 25, 27, 30, 46, 
	48, 51, 55, 58, 67, 71, 72, 74, 
	79, 81, 96, 98, 101, 115, 130, 136, 
	138, 141, 146, 161, 163, 166, 181, 183, 
	186, 190, 197, 199, 202, 207, 212, 214, 
	217, 221, 225, 239, 244, 250, 256, 262, 
	265, 270, 277, 279, 282, 284, 287, 289, 
	292, 295, 297, 300, 302, 305, 307, 314, 
	321, 327, 333, 339, 345, 348, 352, 359, 
	366, 373, 375, 378, 381, 383, 385, 402, 
	419, 434, 440, 442, 445, 450, 465, 467, 
	470, 485, 487, 490, 494, 510, 528, 546, 
	563, 568, 573, 589, 607, 625, 642, 660, 
	678, 695, 706, 718, 730, 742, 751, 762, 
	775, 783, 792, 800, 809, 817, 826, 835, 
	843, 850, 859, 867, 876, 884, 897, 910, 
	922, 934, 946, 958, 967, 977, 990, 1003, 
	1016, 1024, 1033, 1042, 1050, 1058, 1071, 1085, 
	1090, 1092, 1095, 1102, 1104, 1107, 1112, 1116, 
	1125, 1129, 1130, 1135, 1151, 1166, 1172, 1188, 
	1202, 1211, 1216, 1236, 1255, 1265, 1284, 1302, 
	1309, 1318, 1331, 1345, 1359, 1373, 1384, 1397, 
	1412, 1422, 1433, 1443, 1454, 1464, 1475, 1486, 
	1496, 1507, 1517, 1528, 1538, 1553, 1568, 1582, 
	1596, 1610, 1624, 1635, 1647, 1662, 1677, 1692, 
	1702, 1713, 1724, 1734, 1744, 1765, 1786, 1805, 
	1815, 1834, 1852, 1872, 1890, 1910, 1929, 1934, 
	1939, 1957, 1977, 1997, 2016, 2036, 2056, 2075, 
	2088, 2102, 2116, 2130, 2141, 2154, 2169, 2179, 
	2190, 2200, 2211, 2221, 2232, 2243, 2253, 2262, 
	2273, 2283, 2294, 2304, 2319, 2334, 2348, 2362, 
	2376, 2390, 2401, 2413, 2428, 2443, 2458, 2468, 
	2479, 2490, 2500, 2510, 2519, 2529, 2539, 2549, 
	2556, 2565, 2576, 2582, 2589, 2595, 2602, 2608, 
	2615, 2622, 2628, 2635, 2641, 2648, 2654, 2665, 
	2676, 2686, 2696, 2706, 2716, 2723, 2731, 2742, 
	2753, 2764, 2770, 2777, 2784, 2790, 2796, 2813, 
	2830, 2845, 2851, 2867, 2870, 2873, 2876, 2879, 
	2882, 2885, 2888, 2890, 2893, 2896
};

static const short _tsip_machine_parser_header_Referred_By_indicies[] = {
	0, 2, 0, 2, 1, 0, 0, 3, 
	1, 3, 4, 3, 5, 6, 5, 5, 
	7, 5, 5, 5, 5, 8, 5, 8, 
	1, 9, 1, 10, 10, 1, 10, 11, 
	10, 5, 6, 5, 5, 7, 5, 5, 
	5, 5, 8, 5, 8, 1, 12, 1, 
	13, 13, 1, 13, 13, 7, 1, 14, 
	14, 1, 15, 15, 16, 17, 16, 16, 
	16, 16, 1, 15, 15, 17, 1, 18, 
	19, 18, 20, 21, 20, 22, 1, 23, 
	1, 22, 24, 22, 25, 25, 25, 26, 
	26, 25, 25, 25, 25, 25, 25, 1, 
	27, 1, 28, 28, 1, 28, 28, 25, 
	25, 25, 26, 26, 25, 25, 25, 25, 
	25, 25, 1, 29, 30, 29, 31, 31, 
	31, 32, 33, 31, 31, 31, 31, 31, 
	31, 1, 34, 35, 34, 22, 33, 1, 
	36, 1, 37, 37, 1, 37, 37, 22, 
	33, 1, 33, 38, 33, 39, 40, 39, 
	39, 41, 39, 39, 39, 39, 39, 39, 
	1, 42, 1, 43, 43, 1, 43, 44, 
	43, 39, 40, 39, 39, 41, 39, 39, 
	39, 39, 39, 39, 1, 45, 1, 46, 
	46, 1, 46, 46, 40, 1, 47, 48, 
	49, 1, 1, 1, 40, 50, 1, 40, 
	40, 1, 51, 30, 51, 32, 1, 52, 
	53, 52, 22, 1, 54, 1, 55, 55, 
	1, 55, 55, 22, 1, 40, 40, 40, 
	1, 51, 30, 51, 39, 39, 39, 32, 
	39, 39, 39, 39, 39, 39, 1, 57, 
	56, 56, 56, 1, 59, 48, 58, 58, 
	58, 1, 59, 48, 60, 60, 60, 1, 
	59, 48, 61, 61, 61, 1, 59, 48, 
	1, 63, 62, 56, 56, 1, 64, 59, 
	48, 65, 58, 58, 1, 66, 1, 67, 
	68, 1, 69, 1, 70, 71, 1, 72, 
	1, 48, 73, 1, 48, 74, 1, 48, 
	1, 70, 75, 1, 70, 1, 67, 76, 
	1, 67, 1, 64, 59, 48, 77, 60, 
	60, 1, 64, 59, 48, 61, 61, 61, 
	1, 79, 48, 78, 78, 78, 1, 81, 
	48, 80, 80, 80, 1, 81, 48, 82, 
	82, 82, 1, 81, 48, 83, 83, 83, 
	1, 81, 48, 1, 84, 78, 78, 1, 
	64, 81, 48, 85, 80, 80, 1, 64, 
	81, 48, 86, 82, 82, 1, 64, 81, 
	48, 83, 83, 83, 1, 87, 1, 64, 
	88, 1, 64, 89, 1, 64, 1, 63, 
	1, 29, 30, 29, 31, 31, 31, 32, 
	33, 90, 90, 31, 31, 31, 31, 31, 
	31, 1, 29, 30, 29, 31, 31, 31, 
	32, 33, 91, 91, 31, 31, 31, 31, 
	31, 31, 1, 92, 30, 92, 31, 31, 
	31, 32, 93, 31, 31, 31, 31, 31, 
	31, 1, 94, 95, 94, 22, 93, 1, 
	96, 1, 97, 97, 1, 97, 97, 22, 
	93, 1, 98, 99, 98, 39, 100, 39, 
	39, 41, 39, 39, 39, 39, 39, 39, 
	1, 101, 1, 102, 102, 1, 103, 104, 
	103, 39, 100, 39, 39, 41, 39, 39, 
	39, 39, 39, 39, 1, 105, 1, 106, 
	106, 1, 106, 106, 107, 1, 47, 108, 
	48, 108, 108, 108, 49, 108, 1, 1, 
	1, 108, 108, 108, 108, 40, 47, 108, 
	48, 108, 108, 108, 107, 109, 49, 108, 
	1, 1, 1, 108, 108, 108, 108, 40, 
	47, 110, 48, 110, 110, 110, 113, 49, 
	110, 1, 1, 1, 110, 111, 112, 110, 
	112, 40, 47, 110, 114, 110, 110, 110, 
	115, 49, 110, 1, 1, 1, 110, 110, 
	110, 110, 40, 116, 117, 116, 118, 1, 
	119, 120, 119, 121, 1, 47, 110, 48, 
	110, 110, 110, 49, 110, 1, 1, 1, 
	110, 110, 110, 110, 40, 47, 110, 114, 
	110, 110, 122, 123, 49, 110, 1, 1, 
	1, 110, 111, 111, 110, 111, 40, 47, 
	110, 114, 110, 110, 122, 115, 49, 110, 
	1, 1, 1, 110, 111, 111, 110, 111, 
	40, 47, 110, 48, 110, 110, 110, 49, 
	110, 1, 1, 1, 110, 111, 112, 110, 
	112, 40, 47, 110, 114, 110, 110, 124, 
	125, 49, 110, 1, 1, 1, 110, 112, 
	112, 110, 112, 40, 47, 110, 114, 110, 
	110, 124, 115, 49, 110, 1, 1, 1, 
	110, 112, 112, 110, 112, 40, 47, 110, 
	114, 110, 110, 110, 49, 110, 1, 1, 
	1, 110, 111, 112, 110, 112, 40, 47, 
	48, 127, 49, 1, 1, 1, 126, 126, 
	126, 40, 47, 48, 129, 49, 130, 1, 
	1, 1, 128, 128, 128, 40, 47, 48, 
	129, 49, 130, 1, 1, 1, 131, 131, 
	131, 40, 47, 48, 129, 49, 130, 1, 
	1, 1, 132, 132, 132, 40, 47, 48, 
	129, 49, 130, 1, 1, 1, 40, 47, 
	48, 134, 49, 1, 1, 1, 133, 126, 
	126, 40, 47, 48, 135, 129, 49, 130, 
	1, 1, 1, 136, 128, 128, 40, 47, 
	48, 49, 1, 1, 1, 137, 40, 47, 
	48, 138, 49, 1, 1, 1, 139, 40, 
	47, 48, 49, 1, 1, 1, 140, 40, 
	47, 48, 141, 49, 1, 1, 1, 142, 
	40, 47, 48, 49, 1, 1, 1, 143, 
	40, 47, 48, 49, 130, 1, 1, 1, 
	144, 40, 47, 48, 49, 130, 1, 1, 
	1, 145, 40, 47, 48, 49, 130, 1, 
	1, 1, 40, 47, 114, 49, 1, 1, 
	1, 40, 47, 48, 141, 49, 1, 1, 
	1, 146, 40, 47, 48, 141, 49, 1, 
	1, 1, 40, 47, 48, 138, 49, 1, 
	1, 1, 147, 40, 47, 48, 138, 49, 
	1, 1, 1, 40, 47, 48, 135, 129, 
	49, 130, 1, 1, 1, 148, 131, 131, 
	40, 47, 48, 135, 129, 49, 130, 1, 
	1, 1, 132, 132, 132, 40, 47, 48, 
	150, 49, 130, 1, 1, 1, 149, 149, 
	149, 40, 47, 48, 152, 49, 130, 1, 
	1, 1, 151, 151, 151, 40, 47, 48, 
	152, 49, 130, 1, 1, 1, 153, 153, 
	153, 40, 47, 48, 152, 49, 130, 1, 
	1, 1, 154, 154, 154, 40, 47, 48, 
	152, 49, 130, 1, 1, 1, 40, 47, 
	48, 49, 1, 1, 1, 155, 149, 149, 
	40, 47, 48, 135, 152, 49, 130, 1, 
	1, 1, 156, 151, 151, 40, 47, 48, 
	135, 152, 49, 130, 1, 1, 1, 157, 
	153, 153, 40, 47, 48, 135, 152, 49, 
	130, 1, 1, 1, 154, 154, 154, 40, 
	47, 48, 49, 1, 1, 1, 158, 40, 
	47, 48, 135, 49, 1, 1, 1, 159, 
	40, 47, 48, 135, 49, 1, 1, 1, 
	160, 40, 47, 48, 135, 49, 1, 1, 
	1, 40, 47, 48, 134, 49, 1, 1, 
	1, 40, 161, 162, 161, 163, 163, 163, 
	163, 163, 163, 163, 163, 163, 1, 164, 
	165, 164, 163, 163, 163, 166, 163, 163, 
	163, 163, 163, 163, 1, 167, 11, 167, 
	7, 1, 168, 1, 161, 161, 1, 170, 
	171, 172, 1, 1, 1, 169, 173, 1, 
	169, 169, 1, 164, 165, 164, 166, 1, 
	169, 169, 169, 1, 174, 174, 175, 176, 
	175, 175, 175, 175, 1, 174, 174, 176, 
	1, 177, 178, 179, 178, 180, 177, 180, 
	179, 180, 181, 181, 181, 180, 182, 182, 
	181, 181, 181, 181, 181, 181, 177, 183, 
	184, 183, 185, 185, 185, 186, 187, 185, 
	185, 185, 185, 185, 185, 177, 188, 179, 
	188, 180, 187, 177, 189, 179, 189, 190, 
	191, 190, 190, 180, 192, 190, 190, 190, 
	190, 190, 190, 177, 193, 184, 193, 190, 
	190, 190, 186, 190, 190, 190, 190, 190, 
	190, 177, 194, 179, 194, 195, 196, 197, 
	177, 177, 191, 193, 184, 193, 186, 177, 
	196, 179, 196, 198, 195, 198, 198, 196, 
	199, 197, 199, 198, 177, 177, 198, 198, 
	198, 198, 198, 191, 200, 184, 200, 201, 
	195, 201, 201, 202, 203, 197, 201, 177, 
	177, 201, 201, 201, 201, 201, 191, 204, 
	179, 204, 195, 196, 203, 197, 177, 177, 
	191, 205, 179, 205, 206, 207, 206, 206, 
	196, 208, 197, 206, 177, 177, 206, 206, 
	206, 206, 206, 191, 209, 184, 209, 206, 
	195, 206, 206, 202, 197, 206, 177, 177, 
	206, 206, 206, 206, 206, 191, 194, 179, 
	194, 196, 191, 191, 177, 209, 184, 209, 
	195, 202, 197, 177, 177, 191, 194, 179, 
	194, 195, 211, 196, 197, 177, 177, 210, 
	210, 210, 191, 194, 179, 194, 195, 213, 
	196, 197, 207, 177, 177, 212, 212, 212, 
	191, 194, 179, 194, 195, 213, 196, 197, 
	207, 177, 177, 214, 214, 214, 191, 194, 
	179, 194, 195, 213, 196, 197, 207, 177, 
	177, 215, 215, 215, 191, 194, 179, 194, 
	195, 213, 196, 197, 207, 177, 177, 191, 
	194, 179, 194, 195, 217, 196, 197, 177, 
	177, 216, 210, 210, 191, 194, 179, 194, 
	195, 218, 213, 196, 197, 207, 177, 177, 
	219, 212, 212, 191, 194, 179, 194, 195, 
	196, 197, 177, 177, 220, 191, 194, 179, 
	194, 195, 221, 196, 197, 177, 177, 222, 
	191, 194, 179, 194, 195, 196, 197, 177, 
	177, 223, 191, 194, 179, 194, 195, 224, 
	196, 197, 177, 177, 225, 191, 194, 179, 
	194, 195, 196, 197, 177, 177, 226, 191, 
	194, 179, 194, 195, 196, 197, 207, 177, 
	177, 227, 191, 194, 179, 194, 195, 196, 
	197, 207, 177, 177, 228, 191, 194, 179, 
	194, 195, 196, 197, 207, 177, 177, 191, 
	194, 179, 194, 195, 224, 196, 197, 177, 
	177, 229, 191, 194, 179, 194, 195, 224, 
	196, 197, 177, 177, 191, 194, 179, 194, 
	195, 221, 196, 197, 177, 177, 230, 191, 
	194, 179, 194, 195, 221, 196, 197, 177, 
	177, 191, 194, 179, 194, 195, 218, 213, 
	196, 197, 207, 177, 177, 231, 214, 214, 
	191, 194, 179, 194, 195, 218, 213, 196, 
	197, 207, 177, 177, 215, 215, 215, 191, 
	194, 179, 194, 195, 233, 196, 197, 207, 
	177, 177, 232, 232, 232, 191, 194, 179, 
	194, 195, 235, 196, 197, 207, 177, 177, 
	234, 234, 234, 191, 194, 179, 194, 195, 
	235, 196, 197, 207, 177, 177, 236, 236, 
	236, 191, 194, 179, 194, 195, 235, 196, 
	197, 207, 177, 177, 237, 237, 237, 191, 
	194, 179, 194, 195, 235, 196, 197, 207, 
	177, 177, 191, 194, 179, 194, 195, 196, 
	197, 177, 177, 238, 232, 232, 191, 194, 
	179, 194, 195, 218, 235, 196, 197, 207, 
	177, 177, 239, 234, 234, 191, 194, 179, 
	194, 195, 218, 235, 196, 197, 207, 177, 
	177, 240, 236, 236, 191, 194, 179, 194, 
	195, 218, 235, 196, 197, 207, 177, 177, 
	237, 237, 237, 191, 194, 179, 194, 195, 
	196, 197, 177, 177, 241, 191, 194, 179, 
	194, 195, 218, 196, 197, 177, 177, 242, 
	191, 194, 179, 194, 195, 218, 196, 197, 
	177, 177, 243, 191, 194, 179, 194, 195, 
	218, 196, 197, 177, 177, 191, 194, 179, 
	194, 195, 217, 196, 197, 177, 177, 191, 
	200, 184, 200, 201, 195, 201, 201, 202, 
	203, 244, 197, 244, 201, 177, 177, 201, 
	201, 201, 201, 201, 191, 200, 184, 200, 
	201, 195, 201, 201, 202, 203, 245, 197, 
	245, 201, 177, 177, 201, 201, 201, 201, 
	201, 191, 246, 184, 246, 201, 195, 201, 
	201, 202, 247, 197, 201, 177, 177, 201, 
	201, 201, 201, 201, 191, 248, 179, 248, 
	195, 196, 247, 197, 177, 177, 191, 249, 
	179, 249, 206, 250, 206, 206, 196, 208, 
	197, 206, 177, 177, 206, 206, 206, 206, 
	206, 191, 209, 184, 209, 251, 195, 251, 
	251, 251, 202, 197, 251, 177, 177, 251, 
	251, 251, 251, 191, 194, 179, 194, 251, 
	195, 251, 251, 251, 252, 196, 253, 197, 
	251, 177, 177, 251, 251, 251, 251, 191, 
	194, 179, 194, 251, 195, 251, 251, 251, 
	196, 197, 251, 177, 177, 251, 251, 251, 
	251, 191, 194, 179, 194, 254, 195, 254, 
	254, 254, 196, 257, 197, 254, 177, 177, 
	254, 255, 256, 254, 256, 191, 194, 179, 
	194, 254, 258, 254, 254, 254, 259, 196, 
	197, 254, 177, 177, 254, 254, 254, 254, 
	191, 260, 261, 260, 262, 177, 263, 264, 
	263, 265, 177, 194, 179, 194, 254, 195, 
	254, 254, 254, 196, 197, 254, 177, 177, 
	254, 254, 254, 254, 191, 194, 179, 194, 
	254, 258, 254, 254, 266, 267, 196, 197, 
	254, 177, 177, 254, 255, 255, 254, 255, 
	191, 194, 179, 194, 254, 258, 254, 254, 
	266, 259, 196, 197, 254, 177, 177, 254, 
	255, 255, 254, 255, 191, 194, 179, 194, 
	254, 195, 254, 254, 254, 196, 197, 254, 
	177, 177, 254, 255, 256, 254, 256, 191, 
	194, 179, 194, 254, 258, 254, 254, 268, 
	269, 196, 197, 254, 177, 177, 254, 256, 
	256, 254, 256, 191, 194, 179, 194, 254, 
	258, 254, 254, 268, 259, 196, 197, 254, 
	177, 177, 254, 256, 256, 254, 256, 191, 
	194, 179, 194, 254, 258, 254, 254, 254, 
	196, 197, 254, 177, 177, 254, 255, 256, 
	254, 256, 191, 194, 179, 194, 195, 271, 
	196, 197, 177, 177, 270, 270, 270, 191, 
	194, 179, 194, 195, 273, 196, 197, 274, 
	177, 177, 272, 272, 272, 191, 194, 179, 
	194, 195, 273, 196, 197, 274, 177, 177, 
	275, 275, 275, 191, 194, 179, 194, 195, 
	273, 196, 197, 274, 177, 177, 276, 276, 
	276, 191, 194, 179, 194, 195, 273, 196, 
	197, 274, 177, 177, 191, 194, 179, 194, 
	195, 278, 196, 197, 177, 177, 277, 270, 
	270, 191, 194, 179, 194, 195, 279, 273, 
	196, 197, 274, 177, 177, 280, 272, 272, 
	191, 194, 179, 194, 195, 196, 197, 177, 
	177, 281, 191, 194, 179, 194, 195, 282, 
	196, 197, 177, 177, 283, 191, 194, 179, 
	194, 195, 196, 197, 177, 177, 284, 191, 
	194, 179, 194, 195, 285, 196, 197, 177, 
	177, 286, 191, 194, 179, 194, 195, 196, 
	197, 177, 177, 287, 191, 194, 179, 194, 
	195, 196, 197, 274, 177, 177, 288, 191, 
	194, 179, 194, 195, 196, 197, 274, 177, 
	177, 289, 191, 194, 179, 194, 195, 196, 
	197, 274, 177, 177, 191, 194, 179, 194, 
	258, 196, 197, 177, 177, 191, 194, 179, 
	194, 195, 285, 196, 197, 177, 177, 290, 
	191, 194, 179, 194, 195, 285, 196, 197, 
	177, 177, 191, 194, 179, 194, 195, 282, 
	196, 197, 177, 177, 291, 191, 194, 179, 
	194, 195, 282, 196, 197, 177, 177, 191, 
	194, 179, 194, 195, 279, 273, 196, 197, 
	274, 177, 177, 292, 275, 275, 191, 194, 
	179, 194, 195, 279, 273, 196, 197, 274, 
	177, 177, 276, 276, 276, 191, 194, 179, 
	194, 195, 294, 196, 197, 274, 177, 177, 
	293, 293, 293, 191, 194, 179, 194, 195, 
	296, 196, 197, 274, 177, 177, 295, 295, 
	295, 191, 194, 179, 194, 195, 296, 196, 
	197, 274, 177, 177, 297, 297, 297, 191, 
	194, 179, 194, 195, 296, 196, 197, 274, 
	177, 177, 298, 298, 298, 191, 194, 179, 
	194, 195, 296, 196, 197, 274, 177, 177, 
	191, 194, 179, 194, 195, 196, 197, 177, 
	177, 299, 293, 293, 191, 194, 179, 194, 
	195, 279, 296, 196, 197, 274, 177, 177, 
	300, 295, 295, 191, 194, 179, 194, 195, 
	279, 296, 196, 197, 274, 177, 177, 301, 
	297, 297, 191, 194, 179, 194, 195, 279, 
	296, 196, 197, 274, 177, 177, 298, 298, 
	298, 191, 194, 179, 194, 195, 196, 197, 
	177, 177, 302, 191, 194, 179, 194, 195, 
	279, 196, 197, 177, 177, 303, 191, 194, 
	179, 194, 195, 279, 196, 197, 177, 177, 
	304, 191, 194, 179, 194, 195, 279, 196, 
	197, 177, 177, 191, 194, 179, 194, 195, 
	278, 196, 197, 177, 177, 191, 178, 179, 
	178, 306, 180, 305, 305, 305, 177, 178, 
	179, 178, 308, 180, 195, 307, 307, 307, 
	177, 178, 179, 178, 308, 180, 195, 309, 
	309, 309, 177, 178, 179, 178, 308, 180, 
	195, 310, 310, 310, 177, 178, 179, 178, 
	308, 180, 195, 177, 178, 179, 178, 312, 
	180, 311, 305, 305, 177, 178, 179, 178, 
	313, 308, 180, 195, 314, 307, 307, 177, 
	178, 179, 178, 180, 315, 177, 178, 179, 
	178, 316, 180, 317, 177, 178, 179, 178, 
	180, 318, 177, 178, 179, 178, 319, 180, 
	320, 177, 178, 179, 178, 180, 321, 177, 
	178, 179, 178, 180, 195, 322, 177, 178, 
	179, 178, 180, 195, 323, 177, 178, 179, 
	178, 180, 195, 177, 178, 179, 178, 319, 
	180, 324, 177, 178, 179, 178, 319, 180, 
	177, 178, 179, 178, 316, 180, 325, 177, 
	178, 179, 178, 316, 180, 177, 178, 179, 
	178, 313, 308, 180, 195, 326, 309, 309, 
	177, 178, 179, 178, 313, 308, 180, 195, 
	310, 310, 310, 177, 178, 179, 178, 328, 
	180, 195, 327, 327, 327, 177, 178, 179, 
	178, 330, 180, 195, 329, 329, 329, 177, 
	178, 179, 178, 330, 180, 195, 331, 331, 
	331, 177, 178, 179, 178, 330, 180, 195, 
	332, 332, 332, 177, 178, 179, 178, 330, 
	180, 195, 177, 178, 179, 178, 180, 333, 
	327, 327, 177, 178, 179, 178, 313, 330, 
	180, 195, 334, 329, 329, 177, 178, 179, 
	178, 313, 330, 180, 195, 335, 331, 331, 
	177, 178, 179, 178, 313, 330, 180, 195, 
	332, 332, 332, 177, 178, 179, 178, 180, 
	336, 177, 178, 179, 178, 313, 180, 337, 
	177, 178, 179, 178, 313, 180, 338, 177, 
	178, 179, 178, 313, 180, 177, 178, 179, 
	178, 312, 180, 177, 183, 184, 183, 185, 
	185, 185, 186, 187, 339, 339, 185, 185, 
	185, 185, 185, 185, 177, 183, 184, 183, 
	185, 185, 185, 186, 187, 340, 340, 185, 
	185, 185, 185, 185, 185, 177, 341, 184, 
	341, 185, 185, 185, 186, 342, 185, 185, 
	185, 185, 185, 185, 177, 343, 179, 343, 
	180, 342, 177, 344, 179, 344, 190, 345, 
	190, 190, 180, 192, 190, 190, 190, 190, 
	190, 190, 177, 346, 346, 1, 347, 347, 
	1, 348, 348, 1, 349, 349, 1, 350, 
	350, 1, 351, 351, 1, 352, 352, 1, 
	353, 1, 354, 354, 1, 0, 0, 1, 
	1, 0
};

static const short _tsip_machine_parser_header_Referred_By_trans_targs[] = {
	2, 0, 299, 3, 4, 141, 146, 10, 
	151, 5, 6, 7, 8, 9, 11, 12, 
	11, 13, 14, 15, 15, 16, 17, 309, 
	18, 21, 78, 19, 20, 22, 16, 21, 
	17, 26, 22, 23, 24, 25, 27, 42, 
	33, 43, 28, 29, 30, 31, 32, 34, 
	36, 41, 35, 37, 37, 38, 39, 40, 
	44, 77, 45, 48, 46, 47, 49, 64, 
	50, 62, 51, 52, 60, 53, 54, 58, 
	55, 56, 57, 59, 61, 63, 65, 73, 
	66, 69, 67, 68, 70, 71, 72, 74, 
	75, 76, 79, 80, 81, 85, 81, 82, 
	83, 84, 85, 86, 92, 87, 88, 88, 
	89, 90, 91, 92, 93, 94, 95, 99, 
	102, 105, 96, 98, 97, 16, 17, 97, 
	16, 17, 100, 101, 103, 104, 106, 140, 
	107, 110, 120, 108, 109, 111, 127, 112, 
	125, 113, 114, 123, 115, 116, 121, 117, 
	118, 119, 122, 124, 126, 128, 136, 129, 
	132, 130, 131, 133, 134, 135, 137, 138, 
	139, 142, 144, 141, 143, 7, 10, 143, 
	145, 146, 147, 149, 150, 148, 152, 151, 
	153, 154, 154, 16, 155, 156, 294, 157, 
	16, 156, 155, 158, 157, 158, 159, 160, 
	259, 154, 160, 161, 162, 167, 163, 204, 
	164, 163, 162, 165, 164, 165, 166, 168, 
	169, 160, 170, 203, 171, 174, 172, 173, 
	175, 190, 176, 188, 177, 178, 186, 179, 
	180, 184, 181, 182, 183, 185, 187, 189, 
	191, 199, 192, 195, 193, 194, 196, 197, 
	198, 200, 201, 202, 205, 206, 207, 208, 
	207, 208, 209, 210, 211, 212, 213, 217, 
	220, 223, 214, 216, 215, 16, 155, 215, 
	16, 155, 218, 219, 221, 222, 224, 258, 
	225, 228, 238, 226, 227, 229, 245, 230, 
	243, 231, 232, 241, 233, 234, 239, 235, 
	236, 237, 240, 242, 244, 246, 254, 247, 
	250, 248, 249, 251, 252, 253, 255, 256, 
	257, 260, 293, 261, 264, 262, 263, 265, 
	280, 266, 278, 267, 268, 276, 269, 270, 
	274, 271, 272, 273, 275, 277, 279, 281, 
	289, 282, 285, 283, 284, 286, 287, 288, 
	290, 291, 292, 295, 296, 297, 298, 297, 
	298, 211, 300, 301, 302, 303, 304, 305, 
	306, 307, 308
};

static const char _tsip_machine_parser_header_Referred_By_trans_actions[] = {
	0, 0, 0, 0, 0, 1, 1, 0, 
	1, 0, 0, 0, 0, 0, 1, 0, 
	0, 0, 0, 3, 0, 0, 0, 11, 
	0, 1, 1, 0, 0, 9, 9, 0, 
	9, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 9, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 9, 0, 0, 0, 
	0, 0, 1, 1, 1, 0, 0, 1, 
	1, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 22, 22, 22, 7, 
	7, 7, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 5, 5, 5, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 3, 3, 3, 1, 1, 19, 
	19, 0, 19, 0, 3, 3, 0, 0, 
	0, 19, 3, 0, 3, 0, 1, 1, 
	19, 0, 19, 0, 3, 3, 0, 0, 
	0, 19, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 19, 0, 
	3, 13, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 25, 25, 25, 16, 
	16, 16, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 19, 0, 3, 
	13, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0
};

static const int tsip_machine_parser_header_Referred_By_start = 1;
static const int tsip_machine_parser_header_Referred_By_first_final = 309;
static const int tsip_machine_parser_header_Referred_By_error = 0;

static const int tsip_machine_parser_header_Referred_By_en_main = 1;


/* #line 138 "./ragel/tsip_parser_header_Referred_By.rl" */
	(void)(eof);
	(void)(tsip_machine_parser_header_Referred_By_first_final);
	(void)(tsip_machine_parser_header_Referred_By_error);
	(void)(tsip_machine_parser_header_Referred_By_en_main);
	
/* #line 1166 "./src/headers/tsip_header_Referred_By.c" */
	{
	cs = tsip_machine_parser_header_Referred_By_start;
	}

/* #line 143 "./ragel/tsip_parser_header_Referred_By.rl" */
	
/* #line 1173 "./src/headers/tsip_header_Referred_By.c" */
	{
	int _klen;
	unsigned int _trans;
	const char *_acts;
	unsigned int _nacts;
	const char *_keys;

	if ( p == pe )
		goto _test_eof;
	if ( cs == 0 )
		goto _out;
_resume:
	_keys = _tsip_machine_parser_header_Referred_By_trans_keys + _tsip_machine_parser_header_Referred_By_key_offsets[cs];
	_trans = _tsip_machine_parser_header_Referred_By_index_offsets[cs];

	_klen = _tsip_machine_parser_header_Referred_By_single_lengths[cs];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( (*p) < *_mid )
				_upper = _mid - 1;
			else if ( (*p) > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _tsip_machine_parser_header_Referred_By_range_lengths[cs];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( (*p) < _mid[0] )
				_upper = _mid - 2;
			else if ( (*p) > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += ((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	_trans = _tsip_machine_parser_header_Referred_By_indicies[_trans];
	cs = _tsip_machine_parser_header_Referred_By_trans_targs[_trans];

	if ( _tsip_machine_parser_header_Referred_By_trans_actions[_trans] == 0 )
		goto _again;

	_acts = _tsip_machine_parser_header_Referred_By_actions + _tsip_machine_parser_header_Referred_By_trans_actions[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
/* #line 49 "./ragel/tsip_parser_header_Referred_By.rl" */
	{
		tag_start = p;
	}
	break;
	case 1:
/* #line 53 "./ragel/tsip_parser_header_Referred_By.rl" */
	{	
		if(!r_by->uri) /* Only one URI */{
			int len = (int)(p  - tag_start);
			if(r_by && !r_by->uri){
				if((r_by->uri = tsip_uri_parse(tag_start, (tsk_size_t)len)) && r_by->display_name){
					r_by->uri->display_name = tsk_strdup(r_by->display_name);
				}
			}
		}
	}
	break;
	case 2:
/* #line 64 "./ragel/tsip_parser_header_Referred_By.rl" */
	{
		if(!r_by->display_name){
			TSK_PARSER_SET_STRING(r_by->display_name);
			tsk_strunquote(&r_by->display_name);
		}
	}
	break;
	case 3:
/* #line 71 "./ragel/tsip_parser_header_Referred_By.rl" */
	{
		TSK_PARSER_SET_STRING(r_by->cid);
	}
	break;
	case 4:
/* #line 75 "./ragel/tsip_parser_header_Referred_By.rl" */
	{
		TSK_PARSER_ADD_PARAM(TSIP_HEADER_PARAMS(r_by));
	}
	break;
	case 5:
/* #line 79 "./ragel/tsip_parser_header_Referred_By.rl" */
	{
	}
	break;
/* #line 1292 "./src/headers/tsip_header_Referred_By.c" */
		}
	}

_again:
	if ( cs == 0 )
		goto _out;
	if ( ++p != pe )
		goto _resume;
	_test_eof: {}
	_out: {}
	}

/* #line 144 "./ragel/tsip_parser_header_Referred_By.rl" */
	
	if( cs < 
/* #line 1308 "./src/headers/tsip_header_Referred_By.c" */
309
/* #line 145 "./ragel/tsip_parser_header_Referred_By.rl" */
 ){
		TSK_DEBUG_ERROR("Failed to parse 'Referred-By' header.");
		TSK_OBJECT_SAFE_FREE(r_by);
	}
	
	return r_by;
}







//========================================================
//	Referred_By header object definition
//

static tsk_object_t* tsip_header_Referred_By_ctor(tsk_object_t *self, va_list * app)
{
	tsip_header_Referred_By_t *Referred_By = self;
	if(Referred_By){
		const tsip_uri_t* uri = va_arg(*app, const tsip_uri_t*);
		const char* cid = va_arg(*app, const char*);

		TSIP_HEADER(Referred_By)->type = tsip_htype_Referred_By;
		TSIP_HEADER(Referred_By)->serialize = tsip_header_Referred_By_serialize;
		
		Referred_By->uri = tsk_object_ref((void*)uri);
		Referred_By->cid = tsk_strdup(cid);
	}
	else{
		TSK_DEBUG_ERROR("Failed to create new Referred_By header.");
	}
	return self;
}

static tsk_object_t* tsip_header_Referred_By_dtor(tsk_object_t *self)
{
	tsip_header_Referred_By_t *Referred_By = self;
	if(Referred_By){
		TSK_FREE(Referred_By->display_name);
		TSK_OBJECT_SAFE_FREE(Referred_By->uri);
		TSK_FREE(Referred_By->cid);

		TSK_OBJECT_SAFE_FREE(TSIP_HEADER_PARAMS(Referred_By));
	}
	else{
		TSK_DEBUG_ERROR("Null Referred_By header.");
	}

	return self;
}

static const tsk_object_def_t tsip_header_Referred_By_def_s = 
{
	sizeof(tsip_header_Referred_By_t),
	tsip_header_Referred_By_ctor,
	tsip_header_Referred_By_dtor,
	tsk_null
};
const tsk_object_def_t *tsip_header_Referred_By_def_t = &tsip_header_Referred_By_def_s;

