#if HAVE_CRT
#define _CRTDBG_MAP_ALLOC 
#include <stdlib.h> 
#include <crtdbg.h>
#endif //HAVE_CRT

/* #line 1 "./ragel/tsip_parser_header_Via.rl" */
/*
* Copyright (C) 2020, University of the Basque Country (UPV/EHU)
* Contact for licensing options: <licensing-mcpttclient(at)mcopenplatform(dot)com>
*
* The original file was part of Open Source Doubango Framework
* Copyright (C) 2010-2011 Mamadou Diop.
* Copyright (C) 2012 Doubango Telecom <http://doubango.org>
*
* This file is part of Open Source Doubango Framework.
*
* DOUBANGO is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* DOUBANGO is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with DOUBANGO.
*
*/


/**@file tsip_header_Via.c
 * @brief SIP Via/v header as per RFC 3261 subclause 20.42.
 *
 * @author Mamadou Diop <diopmamadou(at)doubango[dot]org>
 *

 */
#include "tinysip/headers/tsip_header_Via.h"

#include "tsk_debug.h"
#include "tsk_memory.h"



/***********************************
*	Ragel state machine.
*/

/* #line 142 "./ragel/tsip_parser_header_Via.rl" */



tsip_header_Via_t* tsip_header_Via_create(const char* proto_name, const char* proto_version, const char* transport, const char* host, uint16_t port)
{
	return tsk_object_new(TSIP_HEADER_VIA_VA_ARGS(proto_name, proto_version, transport, host, port));
}
tsip_header_Via_t* tsip_header_Via_create_null()
{
	return tsip_header_Via_create(tsk_null, tsk_null, tsk_null, tsk_null, 0);
}

int tsip_header_Via_serialize(const tsip_header_t* header, tsk_buffer_t* output)
{
	if(header){
		const tsip_header_Via_t *Via = (const tsip_header_Via_t *)header;
		tsk_istr_t port, rport, ttl;
		int ipv6 = (Via->host && tsk_strcontains(Via->host, tsk_strlen(Via->host), ":"));

		if(Via->port){
			tsk_itoa(Via->port, &port);
		}
		if(Via->rport){
			tsk_itoa(Via->rport, &rport);
		}
		if(Via->ttl){
			tsk_itoa(Via->ttl, &ttl);
		}

		/* SIP/2.0/UDP [::]:1988;test=1234;comp=sigcomp;rport=254;ttl=457;received=192.0.2.101;branch=z9hG4bK1245420841406\r\n" */
		return tsk_buffer_append_2(output, "%s/%s/%s %s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",

			Via->proto_name ? Via->proto_name : "SIP",

			Via->proto_version ? Via->proto_version : "2.0",

			Via->transport ? Via->transport : "UDP",

			ipv6 ? "[" : "",
			Via->host ? Via->host : "127.0.0.1",
			ipv6 ? "]" : "",

			Via->port ? ":" : "",
			Via->port ? port : "",

			Via->maddr ? ";maddr=" : "",
			Via->maddr ? Via->maddr : "",

			Via->sigcomp_id ? ";sigcomp-id=" : "",
			Via->sigcomp_id ? Via->sigcomp_id : "",

			Via->comp ? ";comp=" : "",
			Via->comp ? Via->comp : "",

			Via->rport>=0 ? (Via->rport>0?";rport=":";rport") : "",
			Via->rport>0 ? rport : "",

			Via->ttl>=0 ? (Via->ttl>0?";ttl=":";ttl") : "",
			Via->ttl>0 ? ttl : "",

			Via->received ? ";received=" : "",
			Via->received ? Via->received : "",

			Via->branch ? ";branch=" : "",
			Via->branch ? Via->branch : ""
			);
	}
	return -1;
}

char* tsip_header_Via_get_special_param_value(const tsip_header_t* header, const char* pname)
{
	if(header){
		const tsip_header_Via_t *Via = (const tsip_header_Via_t *)header;
		if(tsk_striequals(pname, "maddr")){
			return tsk_strdup(Via->maddr);
		}
		else if(tsk_striequals(pname, "sigcomp-id")){
			return tsk_strdup(Via->sigcomp_id);
		}
		else if(tsk_striequals(pname, "comp")){
			return tsk_strdup(Via->comp);
		}
		else if(tsk_striequals(pname, "rport")){
			tsk_istr_t rport;
			tsk_itoa(Via->rport, &rport);

			return tsk_strdup(rport);
		}
		else if(tsk_striequals(pname, "received")){
			return tsk_strdup(Via->received);
		}
		else if(tsk_striequals(pname, "branch")){
			return tsk_strdup(Via->branch);
		}
	}
	return tsk_null;
}

tsip_header_Vias_L_t *tsip_header_Via_parse(const char *data, tsk_size_t size)
{
	int cs = 0;
	const char *p = data;
	const char *pe = p + size;
	const char *eof = pe;
	tsip_header_Vias_L_t *hdr_vias = tsk_list_create();
	tsip_header_Via_t *curr_via = tsk_null;
	
	const char *tag_start = tsk_null;

	
/* #line 156 "./src/headers/tsip_header_Via.c" */
static const char _tsip_machine_parser_header_Via_actions[] = {
	0, 1, 0, 1, 2, 1, 3, 1, 
	4, 1, 5, 1, 6, 1, 7, 1, 
	8, 1, 9, 1, 10, 1, 11, 1, 
	13, 1, 14, 1, 16, 2, 1, 0, 
	2, 4, 15, 2, 5, 15, 2, 7, 
	15, 2, 8, 15, 2, 9, 15, 2, 
	10, 15, 2, 11, 15, 2, 12, 13, 
	2, 13, 15, 2, 14, 15, 3, 12, 
	13, 15
};

static const short _tsip_machine_parser_header_Via_key_offsets[] = {
	0, 0, 2, 7, 10, 27, 28, 30, 
	46, 62, 66, 67, 69, 72, 89, 90, 
	92, 108, 124, 128, 129, 131, 134, 151, 
	152, 154, 170, 187, 197, 198, 200, 209, 
	217, 224, 232, 238, 252, 258, 259, 261, 
	266, 271, 272, 274, 278, 285, 290, 291, 
	293, 297, 324, 325, 327, 353, 371, 377, 
	378, 380, 385, 404, 405, 407, 426, 427, 
	429, 432, 440, 441, 443, 448, 449, 455, 
	472, 479, 487, 495, 503, 505, 512, 521, 
	523, 526, 528, 531, 533, 536, 539, 540, 
	543, 544, 547, 548, 557, 566, 574, 582, 
	590, 598, 600, 606, 615, 624, 633, 635, 
	638, 641, 642, 643, 663, 683, 703, 723, 
	743, 761, 767, 768, 770, 775, 794, 795, 
	797, 816, 833, 853, 873, 893, 911, 917, 
	918, 920, 925, 944, 945, 947, 966, 983, 
	1003, 1023, 1043, 1063, 1081, 1087, 1088, 1090, 
	1095, 1116, 1117, 1119, 1140, 1161, 1181, 1202, 
	1221, 1234, 1241, 1252, 1260, 1267, 1273, 1292, 
	1313, 1332, 1353, 1372, 1385, 1398, 1411, 1432, 
	1453, 1474, 1495, 1516, 1537, 1544, 1552, 1560, 
	1568, 1570, 1577, 1586, 1588, 1591, 1593, 1596, 
	1598, 1601, 1604, 1605, 1610, 1613, 1614, 1617, 
	1618, 1627, 1636, 1644, 1652, 1660, 1668, 1670, 
	1676, 1685, 1694, 1703, 1705, 1708, 1711, 1712, 
	1713, 1735, 1755, 1775, 1795, 1815, 1835, 1855, 
	1873, 1879, 1880, 1882, 1887, 1913, 1914, 1916, 
	1942, 1955, 1957, 1960, 1962, 1965, 1967, 1974, 
	1981, 1986, 1989, 1990, 1993, 1994, 2007, 2020, 
	2026, 2033, 2045, 2057, 2069, 2081, 2087, 2093, 
	2106, 2119, 2132, 2134, 2137, 2140, 2141, 2153, 
	2165, 2177, 2178, 2198, 2218, 2238, 2244, 2250, 
	2251, 2253, 2258, 2263, 2264, 2266, 2270, 2277, 
	2297, 2317, 2335, 2341, 2342, 2344, 2349, 2368, 
	2369, 2371, 2390, 2397, 2404, 2409, 2416, 2428, 
	2434, 2442, 2448, 2456, 2462, 2476, 2490, 2504, 
	2512, 2520, 2528, 2536, 2544, 2552, 2559, 2567, 
	2575, 2583, 2585, 2592, 2601, 2603, 2606, 2608, 
	2611, 2613, 2616, 2619, 2620, 2626, 2629, 2630, 
	2633, 2634, 2643, 2652, 2660, 2668, 2676, 2684, 
	2686, 2692, 2701, 2710, 2719, 2721, 2724, 2727, 
	2728, 2729, 2731
};

static const char _tsip_machine_parser_header_Via_trans_keys[] = {
	86, 118, 9, 32, 58, 73, 105, 9, 
	32, 58, 9, 13, 32, 33, 37, 39, 
	126, 42, 43, 45, 46, 48, 57, 65, 
	90, 95, 122, 10, 9, 32, 9, 32, 
	33, 37, 39, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 33, 37, 39, 47, 126, 42, 43, 
	45, 57, 65, 90, 95, 122, 9, 13, 
	32, 47, 10, 9, 32, 9, 32, 47, 
	9, 13, 32, 33, 37, 39, 126, 42, 
	43, 45, 46, 48, 57, 65, 90, 95, 
	122, 10, 9, 32, 9, 32, 33, 37, 
	39, 126, 42, 43, 45, 46, 48, 57, 
	65, 90, 95, 122, 9, 13, 32, 33, 
	37, 39, 47, 126, 42, 43, 45, 57, 
	65, 90, 95, 122, 9, 13, 32, 47, 
	10, 9, 32, 9, 32, 47, 9, 13, 
	32, 33, 37, 39, 126, 42, 43, 45, 
	46, 48, 57, 65, 90, 95, 122, 10, 
	9, 32, 9, 32, 33, 37, 39, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 122, 9, 13, 32, 33, 37, 39, 
	126, 42, 43, 45, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 91, 48, 
	57, 65, 90, 97, 122, 10, 9, 32, 
	9, 32, 91, 48, 57, 65, 90, 97, 
	122, 45, 46, 48, 57, 65, 90, 97, 
	122, 45, 48, 57, 65, 90, 97, 122, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	48, 57, 65, 90, 97, 122, 9, 13, 
	32, 44, 45, 46, 58, 59, 48, 57, 
	65, 90, 97, 122, 9, 13, 32, 44, 
	58, 59, 10, 9, 32, 9, 32, 44, 
	58, 59, 9, 13, 32, 48, 57, 10, 
	9, 32, 9, 32, 48, 57, 9, 13, 
	32, 44, 59, 48, 57, 9, 13, 32, 
	44, 59, 10, 9, 32, 9, 32, 44, 
	59, 9, 13, 32, 33, 37, 39, 66, 
	67, 77, 82, 84, 98, 99, 109, 114, 
	116, 126, 42, 43, 45, 46, 48, 57, 
	65, 90, 95, 122, 10, 9, 32, 9, 
	32, 33, 37, 39, 66, 67, 77, 82, 
	84, 98, 99, 109, 114, 116, 126, 42, 
	43, 45, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 33, 37, 39, 44, 
	59, 61, 126, 42, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 44, 59, 
	61, 10, 9, 32, 9, 32, 44, 59, 
	61, 9, 13, 32, 33, 34, 37, 39, 
	91, 126, 42, 43, 45, 46, 48, 57, 
	65, 90, 95, 122, 10, 9, 32, 9, 
	13, 32, 33, 34, 37, 39, 91, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 122, 10, 9, 32, 9, 32, 34, 
	13, 34, 92, 127, 0, 8, 10, 31, 
	10, 9, 32, 9, 13, 32, 44, 59, 
	10, 0, 9, 11, 12, 14, 127, 9, 
	13, 32, 33, 37, 39, 44, 59, 126, 
	42, 46, 48, 57, 65, 90, 95, 122, 
	58, 48, 57, 65, 70, 97, 102, 58, 
	93, 48, 57, 65, 70, 97, 102, 58, 
	93, 48, 57, 65, 70, 97, 102, 58, 
	93, 48, 57, 65, 70, 97, 102, 58, 
	93, 58, 48, 57, 65, 70, 97, 102, 
	46, 58, 93, 48, 57, 65, 70, 97, 
	102, 48, 57, 46, 48, 57, 48, 57, 
	46, 48, 57, 48, 57, 93, 48, 57, 
	93, 48, 57, 93, 46, 48, 57, 46, 
	46, 48, 57, 46, 46, 58, 93, 48, 
	57, 65, 70, 97, 102, 46, 58, 93, 
	48, 57, 65, 70, 97, 102, 58, 93, 
	48, 57, 65, 70, 97, 102, 58, 93, 
	48, 57, 65, 70, 97, 102, 58, 93, 
	48, 57, 65, 70, 97, 102, 58, 93, 
	48, 57, 65, 70, 97, 102, 58, 93, 
	48, 57, 65, 70, 97, 102, 46, 58, 
	93, 48, 57, 65, 70, 97, 102, 46, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	46, 58, 93, 48, 57, 65, 70, 97, 
	102, 48, 57, 46, 48, 57, 46, 48, 
	57, 46, 58, 9, 13, 32, 33, 37, 
	39, 44, 59, 61, 82, 114, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	65, 97, 126, 42, 46, 48, 57, 66, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 44, 59, 61, 78, 110, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	67, 99, 126, 42, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 44, 59, 61, 72, 104, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	126, 42, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 44, 59, 61, 10, 
	9, 32, 9, 32, 44, 59, 61, 9, 
	13, 32, 33, 34, 37, 39, 91, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 122, 10, 9, 32, 9, 13, 32, 
	33, 34, 37, 39, 91, 126, 42, 43, 
	45, 46, 48, 57, 65, 90, 95, 122, 
	9, 13, 32, 33, 37, 39, 44, 59, 
	126, 42, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 33, 37, 39, 44, 
	59, 61, 79, 111, 126, 42, 46, 48, 
	57, 65, 90, 95, 122, 9, 13, 32, 
	33, 37, 39, 44, 59, 61, 77, 109, 
	126, 42, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 33, 37, 39, 44, 
	59, 61, 80, 112, 126, 42, 46, 48, 
	57, 65, 90, 95, 122, 9, 13, 32, 
	33, 37, 39, 44, 59, 61, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 44, 59, 61, 10, 9, 32, 
	9, 32, 44, 59, 61, 9, 13, 32, 
	33, 34, 37, 39, 91, 126, 42, 43, 
	45, 46, 48, 57, 65, 90, 95, 122, 
	10, 9, 32, 9, 13, 32, 33, 34, 
	37, 39, 91, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 33, 37, 39, 44, 59, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	65, 97, 126, 42, 46, 48, 57, 66, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 44, 59, 61, 68, 100, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	68, 100, 126, 42, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 44, 59, 61, 82, 114, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	126, 42, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 44, 59, 61, 10, 
	9, 32, 9, 32, 44, 59, 61, 9, 
	13, 32, 33, 34, 37, 39, 91, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 96, 97, 122, 10, 9, 32, 9, 
	13, 32, 33, 34, 37, 39, 91, 126, 
	42, 43, 45, 46, 48, 57, 65, 90, 
	95, 96, 97, 122, 9, 13, 32, 33, 
	37, 39, 44, 45, 46, 59, 126, 42, 
	43, 48, 57, 65, 90, 95, 96, 97, 
	122, 9, 13, 32, 33, 37, 39, 44, 
	45, 59, 126, 42, 46, 48, 57, 65, 
	90, 95, 96, 97, 122, 9, 13, 32, 
	33, 37, 39, 44, 45, 46, 59, 126, 
	42, 43, 48, 57, 65, 90, 95, 96, 
	97, 122, 9, 13, 32, 33, 37, 39, 
	44, 59, 126, 42, 46, 48, 57, 65, 
	90, 95, 96, 97, 122, 9, 13, 32, 
	44, 45, 46, 59, 48, 57, 65, 90, 
	97, 122, 45, 48, 57, 65, 90, 97, 
	122, 9, 13, 32, 44, 59, 48, 57, 
	65, 90, 97, 122, 45, 46, 48, 57, 
	65, 90, 97, 122, 45, 48, 57, 65, 
	90, 97, 122, 48, 57, 65, 90, 97, 
	122, 9, 13, 32, 33, 37, 39, 44, 
	59, 126, 42, 46, 48, 57, 65, 90, 
	95, 96, 97, 122, 9, 13, 32, 33, 
	37, 39, 44, 45, 46, 59, 126, 42, 
	43, 48, 57, 65, 90, 95, 96, 97, 
	122, 9, 13, 32, 33, 37, 39, 44, 
	59, 126, 42, 46, 48, 57, 65, 90, 
	95, 96, 97, 122, 9, 13, 32, 33, 
	37, 39, 44, 45, 46, 59, 126, 42, 
	43, 48, 57, 65, 90, 95, 96, 97, 
	122, 9, 13, 32, 33, 37, 39, 44, 
	59, 126, 42, 46, 48, 57, 65, 90, 
	95, 96, 97, 122, 9, 13, 32, 44, 
	45, 46, 59, 48, 57, 65, 90, 97, 
	122, 9, 13, 32, 44, 45, 46, 59, 
	48, 57, 65, 90, 97, 122, 9, 13, 
	32, 44, 45, 46, 59, 48, 57, 65, 
	90, 97, 122, 9, 13, 32, 33, 37, 
	39, 44, 45, 46, 59, 126, 42, 43, 
	48, 57, 65, 90, 95, 96, 97, 122, 
	9, 13, 32, 33, 37, 39, 44, 45, 
	46, 59, 126, 42, 43, 48, 57, 65, 
	90, 95, 96, 97, 122, 9, 13, 32, 
	33, 37, 39, 44, 45, 46, 59, 126, 
	42, 43, 48, 57, 65, 90, 95, 96, 
	97, 122, 9, 13, 32, 33, 37, 39, 
	44, 45, 46, 59, 126, 42, 43, 48, 
	57, 65, 90, 95, 96, 97, 122, 9, 
	13, 32, 33, 37, 39, 44, 45, 46, 
	59, 126, 42, 43, 48, 57, 65, 90, 
	95, 96, 97, 122, 9, 13, 32, 33, 
	37, 39, 44, 45, 46, 59, 126, 42, 
	43, 48, 57, 65, 90, 95, 96, 97, 
	122, 58, 48, 57, 65, 70, 97, 102, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	58, 93, 48, 57, 65, 70, 97, 102, 
	58, 93, 58, 48, 57, 65, 70, 97, 
	102, 46, 58, 93, 48, 57, 65, 70, 
	97, 102, 48, 57, 46, 48, 57, 48, 
	57, 46, 48, 57, 48, 57, 93, 48, 
	57, 93, 48, 57, 93, 9, 13, 32, 
	44, 59, 46, 48, 57, 46, 46, 48, 
	57, 46, 46, 58, 93, 48, 57, 65, 
	70, 97, 102, 46, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 46, 58, 93, 48, 
	57, 65, 70, 97, 102, 46, 58, 93, 
	48, 57, 65, 70, 97, 102, 46, 58, 
	93, 48, 57, 65, 70, 97, 102, 48, 
	57, 46, 48, 57, 46, 48, 57, 46, 
	58, 9, 13, 32, 33, 37, 39, 44, 
	59, 61, 69, 80, 101, 112, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	67, 99, 126, 42, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 44, 59, 61, 69, 101, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	73, 105, 126, 42, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 44, 59, 61, 86, 118, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	69, 101, 126, 42, 46, 48, 57, 65, 
	90, 95, 122, 9, 13, 32, 33, 37, 
	39, 44, 59, 61, 68, 100, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 33, 37, 39, 44, 59, 61, 
	126, 42, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 44, 59, 61, 10, 
	9, 32, 9, 32, 44, 59, 61, 9, 
	13, 32, 33, 34, 37, 39, 58, 91, 
	126, 42, 43, 45, 46, 48, 57, 65, 
	70, 71, 90, 95, 96, 97, 102, 103, 
	122, 10, 9, 32, 9, 13, 32, 33, 
	34, 37, 39, 58, 91, 126, 42, 43, 
	45, 46, 48, 57, 65, 70, 71, 90, 
	95, 96, 97, 102, 103, 122, 9, 13, 
	32, 44, 46, 58, 59, 48, 57, 65, 
	70, 97, 102, 48, 57, 46, 48, 57, 
	48, 57, 46, 48, 57, 48, 57, 9, 
	13, 32, 44, 59, 48, 57, 9, 13, 
	32, 44, 59, 48, 57, 9, 13, 32, 
	44, 59, 46, 48, 57, 46, 46, 48, 
	57, 46, 9, 13, 32, 44, 46, 58, 
	59, 48, 57, 65, 70, 97, 102, 9, 
	13, 32, 44, 46, 58, 59, 48, 57, 
	65, 70, 97, 102, 9, 13, 32, 44, 
	58, 59, 58, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 44, 58, 59, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	44, 58, 59, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 44, 58, 59, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	44, 58, 59, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 44, 58, 59, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	44, 46, 58, 59, 48, 57, 65, 70, 
	97, 102, 9, 13, 32, 44, 46, 58, 
	59, 48, 57, 65, 70, 97, 102, 9, 
	13, 32, 44, 46, 58, 59, 48, 57, 
	65, 70, 97, 102, 48, 57, 46, 48, 
	57, 46, 48, 57, 46, 9, 13, 32, 
	44, 58, 59, 48, 57, 65, 70, 97, 
	102, 9, 13, 32, 44, 58, 59, 48, 
	57, 65, 70, 97, 102, 9, 13, 32, 
	44, 58, 59, 48, 57, 65, 70, 97, 
	102, 58, 9, 13, 32, 33, 37, 39, 
	44, 59, 61, 79, 111, 126, 42, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 33, 37, 39, 44, 59, 61, 82, 
	114, 126, 42, 46, 48, 57, 65, 90, 
	95, 122, 9, 13, 32, 33, 37, 39, 
	44, 59, 61, 84, 116, 126, 42, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 44, 59, 61, 9, 13, 32, 44, 
	59, 61, 10, 9, 32, 9, 32, 44, 
	59, 61, 9, 13, 32, 48, 57, 10, 
	9, 32, 9, 32, 48, 57, 9, 13, 
	32, 44, 59, 48, 57, 9, 13, 32, 
	33, 37, 39, 44, 59, 61, 84, 116, 
	126, 42, 46, 48, 57, 65, 90, 95, 
	122, 9, 13, 32, 33, 37, 39, 44, 
	59, 61, 76, 108, 126, 42, 46, 48, 
	57, 65, 90, 95, 122, 9, 13, 32, 
	33, 37, 39, 44, 59, 61, 126, 42, 
	46, 48, 57, 65, 90, 95, 122, 9, 
	13, 32, 44, 59, 61, 10, 9, 32, 
	9, 32, 44, 59, 61, 9, 13, 32, 
	33, 34, 37, 39, 91, 126, 42, 43, 
	45, 46, 48, 57, 65, 90, 95, 122, 
	10, 9, 32, 9, 13, 32, 33, 34, 
	37, 39, 91, 126, 42, 43, 45, 46, 
	48, 57, 65, 90, 95, 122, 9, 13, 
	32, 44, 59, 48, 57, 9, 13, 32, 
	44, 59, 48, 57, 9, 13, 32, 44, 
	59, 45, 48, 57, 65, 90, 97, 122, 
	9, 13, 32, 44, 58, 59, 48, 57, 
	65, 90, 97, 122, 48, 57, 65, 90, 
	97, 122, 45, 46, 48, 57, 65, 90, 
	97, 122, 48, 57, 65, 90, 97, 122, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	48, 57, 65, 90, 97, 122, 9, 13, 
	32, 44, 45, 46, 58, 59, 48, 57, 
	65, 90, 97, 122, 9, 13, 32, 44, 
	45, 46, 58, 59, 48, 57, 65, 90, 
	97, 122, 9, 13, 32, 44, 45, 46, 
	58, 59, 48, 57, 65, 90, 97, 122, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	45, 46, 48, 57, 65, 90, 97, 122, 
	58, 48, 57, 65, 70, 97, 102, 58, 
	93, 48, 57, 65, 70, 97, 102, 58, 
	93, 48, 57, 65, 70, 97, 102, 58, 
	93, 48, 57, 65, 70, 97, 102, 58, 
	93, 58, 48, 57, 65, 70, 97, 102, 
	46, 58, 93, 48, 57, 65, 70, 97, 
	102, 48, 57, 46, 48, 57, 48, 57, 
	46, 48, 57, 48, 57, 93, 48, 57, 
	93, 48, 57, 93, 9, 13, 32, 44, 
	58, 59, 46, 48, 57, 46, 46, 48, 
	57, 46, 46, 58, 93, 48, 57, 65, 
	70, 97, 102, 46, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 58, 93, 48, 57, 
	65, 70, 97, 102, 46, 58, 93, 48, 
	57, 65, 70, 97, 102, 46, 58, 93, 
	48, 57, 65, 70, 97, 102, 46, 58, 
	93, 48, 57, 65, 70, 97, 102, 48, 
	57, 46, 48, 57, 46, 48, 57, 46, 
	58, 65, 97, 0
};

static const char _tsip_machine_parser_header_Via_single_lengths[] = {
	0, 2, 5, 3, 7, 1, 2, 6, 
	8, 4, 1, 2, 3, 7, 1, 2, 
	6, 8, 4, 1, 2, 3, 7, 1, 
	2, 6, 7, 4, 1, 2, 3, 2, 
	1, 2, 0, 8, 6, 1, 2, 5, 
	3, 1, 2, 2, 5, 5, 1, 2, 
	4, 17, 1, 2, 16, 10, 6, 1, 
	2, 5, 9, 1, 2, 9, 1, 2, 
	3, 4, 1, 2, 5, 1, 0, 9, 
	1, 2, 2, 2, 2, 1, 3, 0, 
	1, 0, 1, 0, 1, 1, 1, 1, 
	1, 1, 1, 3, 3, 2, 2, 2, 
	2, 2, 0, 3, 3, 3, 0, 1, 
	1, 1, 1, 12, 12, 12, 12, 12, 
	10, 6, 1, 2, 5, 9, 1, 2, 
	9, 9, 12, 12, 12, 10, 6, 1, 
	2, 5, 9, 1, 2, 9, 9, 12, 
	12, 12, 12, 10, 6, 1, 2, 5, 
	9, 1, 2, 9, 11, 10, 11, 9, 
	7, 1, 5, 2, 1, 0, 9, 11, 
	9, 11, 9, 7, 7, 7, 11, 11, 
	11, 11, 11, 11, 1, 2, 2, 2, 
	2, 1, 3, 0, 1, 0, 1, 0, 
	1, 1, 1, 5, 1, 1, 1, 1, 
	3, 3, 2, 2, 2, 2, 2, 0, 
	3, 3, 3, 0, 1, 1, 1, 1, 
	14, 12, 12, 12, 12, 12, 12, 10, 
	6, 1, 2, 5, 10, 1, 2, 10, 
	7, 0, 1, 0, 1, 0, 5, 5, 
	5, 1, 1, 1, 1, 7, 7, 6, 
	1, 6, 6, 6, 6, 6, 0, 7, 
	7, 7, 0, 1, 1, 1, 6, 6, 
	6, 1, 12, 12, 12, 6, 6, 1, 
	2, 5, 3, 1, 2, 2, 5, 12, 
	12, 10, 6, 1, 2, 5, 9, 1, 
	2, 9, 5, 5, 5, 1, 6, 0, 
	2, 0, 2, 0, 8, 8, 8, 2, 
	2, 2, 2, 2, 2, 1, 2, 2, 
	2, 2, 1, 3, 0, 1, 0, 1, 
	0, 1, 1, 1, 6, 1, 1, 1, 
	1, 3, 3, 2, 2, 2, 2, 2, 
	0, 3, 3, 3, 0, 1, 1, 1, 
	1, 2, 0
};

static const char _tsip_machine_parser_header_Via_range_lengths[] = {
	0, 0, 0, 0, 5, 0, 0, 5, 
	4, 0, 0, 0, 0, 5, 0, 0, 
	5, 4, 0, 0, 0, 0, 5, 0, 
	0, 5, 5, 3, 0, 0, 3, 3, 
	3, 3, 3, 3, 0, 0, 0, 0, 
	1, 0, 0, 1, 1, 0, 0, 0, 
	0, 5, 0, 0, 5, 4, 0, 0, 
	0, 0, 5, 0, 0, 5, 0, 0, 
	0, 2, 0, 0, 0, 0, 3, 4, 
	3, 3, 3, 3, 0, 3, 3, 1, 
	1, 1, 1, 1, 1, 1, 0, 1, 
	0, 1, 0, 3, 3, 3, 3, 3, 
	3, 0, 3, 3, 3, 3, 1, 1, 
	1, 0, 0, 4, 4, 4, 4, 4, 
	4, 0, 0, 0, 0, 5, 0, 0, 
	5, 4, 4, 4, 4, 4, 0, 0, 
	0, 0, 5, 0, 0, 5, 4, 4, 
	4, 4, 4, 4, 0, 0, 0, 0, 
	6, 0, 0, 6, 5, 5, 5, 5, 
	3, 3, 3, 3, 3, 3, 5, 5, 
	5, 5, 5, 3, 3, 3, 5, 5, 
	5, 5, 5, 5, 3, 3, 3, 3, 
	0, 3, 3, 1, 1, 1, 1, 1, 
	1, 1, 0, 0, 1, 0, 1, 0, 
	3, 3, 3, 3, 3, 3, 0, 3, 
	3, 3, 3, 1, 1, 1, 0, 0, 
	4, 4, 4, 4, 4, 4, 4, 4, 
	0, 0, 0, 0, 8, 0, 0, 8, 
	3, 1, 1, 1, 1, 1, 1, 1, 
	0, 1, 0, 1, 0, 3, 3, 0, 
	3, 3, 3, 3, 3, 0, 3, 3, 
	3, 3, 1, 1, 1, 0, 3, 3, 
	3, 0, 4, 4, 4, 0, 0, 0, 
	0, 0, 1, 0, 0, 1, 1, 4, 
	4, 4, 0, 0, 0, 0, 5, 0, 
	0, 5, 1, 1, 0, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 3, 3, 3, 3, 3, 3, 3, 
	3, 0, 3, 3, 1, 1, 1, 1, 
	1, 1, 1, 0, 0, 1, 0, 1, 
	0, 3, 3, 3, 3, 3, 3, 0, 
	3, 3, 3, 3, 1, 1, 1, 0, 
	0, 0, 0
};

static const short _tsip_machine_parser_header_Via_index_offsets[] = {
	0, 0, 3, 9, 13, 26, 28, 31, 
	43, 56, 61, 63, 66, 70, 83, 85, 
	88, 100, 113, 118, 120, 123, 127, 140, 
	142, 145, 157, 170, 178, 180, 183, 190, 
	196, 201, 207, 211, 223, 230, 232, 235, 
	241, 246, 248, 251, 255, 262, 268, 270, 
	273, 278, 301, 303, 306, 328, 343, 350, 
	352, 355, 361, 376, 378, 381, 396, 398, 
	401, 405, 412, 414, 417, 423, 425, 429, 
	443, 448, 454, 460, 466, 469, 474, 481, 
	483, 486, 488, 491, 493, 496, 499, 501, 
	504, 506, 509, 511, 518, 525, 531, 537, 
	543, 549, 552, 556, 563, 570, 577, 579, 
	582, 585, 587, 589, 606, 623, 640, 657, 
	674, 689, 696, 698, 701, 707, 722, 724, 
	727, 742, 756, 773, 790, 807, 822, 829, 
	831, 834, 840, 855, 857, 860, 875, 889, 
	906, 923, 940, 957, 972, 979, 981, 984, 
	990, 1006, 1008, 1011, 1027, 1044, 1060, 1077, 
	1092, 1103, 1108, 1117, 1123, 1128, 1132, 1147, 
	1164, 1179, 1196, 1211, 1222, 1233, 1244, 1261, 
	1278, 1295, 1312, 1329, 1346, 1351, 1357, 1363, 
	1369, 1372, 1377, 1384, 1386, 1389, 1391, 1394, 
	1396, 1399, 1402, 1404, 1410, 1413, 1415, 1418, 
	1420, 1427, 1434, 1440, 1446, 1452, 1458, 1461, 
	1465, 1472, 1479, 1486, 1488, 1491, 1494, 1496, 
	1498, 1517, 1534, 1551, 1568, 1585, 1602, 1619, 
	1634, 1641, 1643, 1646, 1652, 1671, 1673, 1676, 
	1695, 1706, 1708, 1711, 1713, 1716, 1718, 1725, 
	1732, 1738, 1741, 1743, 1746, 1748, 1759, 1770, 
	1777, 1782, 1792, 1802, 1812, 1822, 1829, 1833, 
	1844, 1855, 1866, 1868, 1871, 1874, 1876, 1886, 
	1896, 1906, 1908, 1925, 1942, 1959, 1966, 1973, 
	1975, 1978, 1984, 1989, 1991, 1994, 1998, 2005, 
	2022, 2039, 2054, 2061, 2063, 2066, 2072, 2087, 
	2089, 2092, 2107, 2114, 2121, 2127, 2132, 2142, 
	2146, 2152, 2156, 2162, 2166, 2178, 2190, 2202, 
	2208, 2214, 2220, 2226, 2232, 2238, 2243, 2249, 
	2255, 2261, 2264, 2269, 2276, 2278, 2281, 2283, 
	2286, 2288, 2291, 2294, 2296, 2303, 2306, 2308, 
	2311, 2313, 2320, 2327, 2333, 2339, 2345, 2351, 
	2354, 2358, 2365, 2372, 2379, 2381, 2384, 2387, 
	2389, 2391, 2394
};

static const short _tsip_machine_parser_header_Via_indicies[] = {
	0, 0, 1, 2, 2, 3, 4, 4, 
	1, 2, 2, 3, 1, 3, 5, 3, 
	6, 6, 6, 6, 6, 6, 6, 6, 
	6, 1, 7, 1, 8, 8, 1, 8, 
	8, 6, 6, 6, 6, 6, 6, 6, 
	6, 6, 1, 9, 10, 9, 11, 11, 
	11, 12, 11, 11, 11, 11, 11, 1, 
	13, 14, 13, 15, 1, 16, 1, 17, 
	17, 1, 17, 17, 15, 1, 15, 18, 
	15, 19, 19, 19, 19, 19, 19, 19, 
	19, 19, 1, 20, 1, 21, 21, 1, 
	21, 21, 19, 19, 19, 19, 19, 19, 
	19, 19, 19, 1, 22, 23, 22, 24, 
	24, 24, 25, 24, 24, 24, 24, 24, 
	1, 26, 27, 26, 28, 1, 29, 1, 
	30, 30, 1, 30, 30, 28, 1, 28, 
	31, 28, 32, 32, 32, 32, 32, 32, 
	32, 32, 32, 1, 33, 1, 34, 34, 
	1, 34, 34, 32, 32, 32, 32, 32, 
	32, 32, 32, 32, 1, 35, 36, 35, 
	37, 37, 37, 37, 37, 37, 37, 37, 
	37, 1, 38, 39, 38, 42, 40, 41, 
	41, 1, 43, 1, 44, 44, 1, 44, 
	44, 42, 40, 41, 41, 1, 45, 46, 
	47, 48, 48, 1, 45, 48, 48, 48, 
	1, 45, 49, 48, 48, 48, 1, 48, 
	50, 50, 1, 51, 52, 51, 53, 54, 
	55, 56, 57, 50, 50, 50, 1, 58, 
	59, 58, 3, 60, 61, 1, 62, 1, 
	63, 63, 1, 63, 63, 3, 60, 61, 
	1, 60, 64, 60, 65, 1, 66, 1, 
	67, 67, 1, 67, 67, 65, 1, 68, 
	69, 68, 70, 72, 71, 1, 73, 74, 
	73, 3, 61, 1, 75, 1, 76, 76, 
	1, 76, 76, 3, 61, 1, 61, 77, 
	61, 78, 78, 78, 79, 80, 81, 82, 
	83, 79, 80, 81, 82, 83, 78, 78, 
	78, 78, 78, 78, 1, 84, 1, 85, 
	85, 1, 85, 85, 78, 78, 78, 79, 
	80, 81, 82, 83, 79, 80, 81, 82, 
	83, 78, 78, 78, 78, 78, 78, 1, 
	86, 87, 86, 88, 88, 88, 89, 90, 
	91, 88, 88, 88, 88, 88, 1, 92, 
	93, 92, 3, 61, 91, 1, 94, 1, 
	95, 95, 1, 95, 95, 3, 61, 91, 
	1, 91, 96, 91, 97, 98, 97, 97, 
	99, 97, 97, 97, 97, 97, 97, 1, 
	100, 1, 101, 101, 1, 101, 102, 101, 
	97, 98, 97, 97, 99, 97, 97, 97, 
	97, 97, 97, 1, 103, 1, 104, 104, 
	1, 104, 104, 98, 1, 105, 106, 107, 
	1, 1, 1, 98, 108, 1, 98, 98, 
	1, 109, 87, 109, 89, 90, 1, 110, 
	1, 98, 98, 98, 1, 109, 87, 109, 
	97, 97, 97, 89, 90, 97, 97, 97, 
	97, 97, 1, 112, 111, 111, 111, 1, 
	114, 106, 113, 113, 113, 1, 114, 106, 
	115, 115, 115, 1, 114, 106, 116, 116, 
	116, 1, 114, 106, 1, 118, 117, 111, 
	111, 1, 119, 114, 106, 120, 113, 113, 
	1, 121, 1, 122, 123, 1, 124, 1, 
	125, 126, 1, 127, 1, 106, 128, 1, 
	106, 129, 1, 106, 1, 125, 130, 1, 
	125, 1, 122, 131, 1, 122, 1, 119, 
	114, 106, 132, 115, 115, 1, 119, 114, 
	106, 116, 116, 116, 1, 134, 106, 133, 
	133, 133, 1, 136, 106, 135, 135, 135, 
	1, 136, 106, 137, 137, 137, 1, 136, 
	106, 138, 138, 138, 1, 136, 106, 1, 
	139, 133, 133, 1, 119, 136, 106, 140, 
	135, 135, 1, 119, 136, 106, 141, 137, 
	137, 1, 119, 136, 106, 138, 138, 138, 
	1, 142, 1, 119, 143, 1, 119, 144, 
	1, 119, 1, 118, 1, 86, 87, 86, 
	88, 88, 88, 89, 90, 91, 145, 145, 
	88, 88, 88, 88, 88, 1, 86, 87, 
	86, 88, 88, 88, 89, 90, 91, 146, 
	146, 88, 88, 88, 88, 88, 1, 86, 
	87, 86, 88, 88, 88, 89, 90, 91, 
	147, 147, 88, 88, 88, 88, 88, 1, 
	86, 87, 86, 88, 88, 88, 89, 90, 
	91, 148, 148, 88, 88, 88, 88, 88, 
	1, 86, 87, 86, 88, 88, 88, 89, 
	90, 91, 149, 149, 88, 88, 88, 88, 
	88, 1, 150, 87, 150, 88, 88, 88, 
	89, 90, 151, 88, 88, 88, 88, 88, 
	1, 152, 153, 152, 3, 61, 151, 1, 
	154, 1, 155, 155, 1, 155, 155, 3, 
	61, 151, 1, 151, 156, 151, 157, 98, 
	157, 157, 99, 157, 157, 157, 157, 157, 
	157, 1, 158, 1, 159, 159, 1, 159, 
	102, 159, 157, 98, 157, 157, 99, 157, 
	157, 157, 157, 157, 157, 1, 160, 161, 
	160, 162, 162, 162, 163, 164, 162, 162, 
	162, 162, 162, 1, 86, 87, 86, 88, 
	88, 88, 89, 90, 91, 165, 165, 88, 
	88, 88, 88, 88, 1, 86, 87, 86, 
	88, 88, 88, 89, 90, 91, 166, 166, 
	88, 88, 88, 88, 88, 1, 86, 87, 
	86, 88, 88, 88, 89, 90, 91, 167, 
	167, 88, 88, 88, 88, 88, 1, 168, 
	87, 168, 88, 88, 88, 89, 90, 169, 
	88, 88, 88, 88, 88, 1, 170, 171, 
	170, 3, 61, 169, 1, 172, 1, 173, 
	173, 1, 173, 173, 3, 61, 169, 1, 
	169, 174, 169, 175, 98, 175, 175, 99, 
	175, 175, 175, 175, 175, 175, 1, 176, 
	1, 177, 177, 1, 177, 102, 177, 175, 
	98, 175, 175, 99, 175, 175, 175, 175, 
	175, 175, 1, 178, 179, 178, 180, 180, 
	180, 181, 182, 180, 180, 180, 180, 180, 
	1, 86, 87, 86, 88, 88, 88, 89, 
	90, 91, 183, 183, 88, 88, 88, 88, 
	88, 1, 86, 87, 86, 88, 88, 88, 
	89, 90, 91, 184, 184, 88, 88, 88, 
	88, 88, 1, 86, 87, 86, 88, 88, 
	88, 89, 90, 91, 185, 185, 88, 88, 
	88, 88, 88, 1, 86, 87, 86, 88, 
	88, 88, 89, 90, 91, 186, 186, 88, 
	88, 88, 88, 88, 1, 187, 87, 187, 
	88, 88, 88, 89, 90, 188, 88, 88, 
	88, 88, 88, 1, 189, 190, 189, 3, 
	61, 188, 1, 191, 1, 192, 192, 1, 
	192, 192, 3, 61, 188, 1, 188, 193, 
	188, 97, 98, 97, 97, 196, 97, 97, 
	97, 194, 195, 97, 195, 1, 197, 1, 
	198, 198, 1, 198, 102, 198, 97, 98, 
	97, 97, 196, 97, 97, 97, 194, 195, 
	97, 195, 1, 109, 87, 109, 97, 97, 
	97, 89, 199, 200, 90, 97, 97, 201, 
	202, 97, 202, 1, 109, 87, 109, 97, 
	97, 97, 89, 199, 90, 97, 97, 202, 
	202, 97, 202, 1, 109, 87, 109, 97, 
	97, 97, 89, 199, 203, 90, 97, 97, 
	202, 202, 97, 202, 1, 109, 87, 109, 
	97, 97, 97, 89, 90, 97, 97, 202, 
	204, 97, 204, 1, 205, 206, 205, 207, 
	208, 209, 210, 204, 204, 204, 1, 208, 
	204, 204, 204, 1, 205, 206, 205, 207, 
	210, 211, 204, 204, 1, 212, 213, 211, 
	211, 211, 1, 212, 211, 211, 211, 1, 
	211, 204, 204, 1, 109, 87, 109, 97, 
	97, 97, 89, 90, 97, 97, 214, 204, 
	97, 204, 1, 109, 87, 109, 97, 97, 
	97, 89, 199, 215, 90, 97, 97, 216, 
	202, 97, 202, 1, 109, 87, 109, 97, 
	97, 97, 89, 90, 97, 97, 217, 204, 
	97, 204, 1, 109, 87, 109, 97, 97, 
	97, 89, 199, 218, 90, 97, 97, 219, 
	202, 97, 202, 1, 109, 87, 109, 97, 
	97, 97, 89, 90, 97, 97, 220, 204, 
	97, 204, 1, 205, 206, 205, 207, 212, 
	213, 210, 221, 211, 211, 1, 205, 206, 
	205, 207, 212, 213, 210, 222, 211, 211, 
	1, 205, 206, 205, 207, 212, 213, 210, 
	211, 211, 211, 1, 109, 87, 109, 97, 
	97, 97, 89, 199, 218, 90, 97, 97, 
	223, 202, 97, 202, 1, 109, 87, 109, 
	97, 97, 97, 89, 199, 218, 90, 97, 
	97, 202, 202, 97, 202, 1, 109, 87, 
	109, 97, 97, 97, 89, 199, 215, 90, 
	97, 97, 224, 202, 97, 202, 1, 109, 
	87, 109, 97, 97, 97, 89, 199, 215, 
	90, 97, 97, 202, 202, 97, 202, 1, 
	109, 87, 109, 97, 97, 97, 89, 199, 
	200, 90, 97, 97, 225, 202, 97, 202, 
	1, 109, 87, 109, 97, 97, 97, 89, 
	199, 200, 90, 97, 97, 202, 202, 97, 
	202, 1, 227, 226, 226, 226, 1, 229, 
	230, 228, 228, 228, 1, 229, 230, 231, 
	231, 231, 1, 229, 230, 232, 232, 232, 
	1, 229, 230, 1, 234, 233, 226, 226, 
	1, 235, 229, 230, 236, 228, 228, 1, 
	237, 1, 238, 239, 1, 240, 1, 241, 
	242, 1, 243, 1, 230, 244, 1, 230, 
	245, 1, 230, 1, 205, 206, 205, 207, 
	210, 1, 241, 246, 1, 241, 1, 238, 
	247, 1, 238, 1, 235, 229, 230, 248, 
	231, 231, 1, 235, 229, 230, 232, 232, 
	232, 1, 250, 230, 249, 249, 249, 1, 
	252, 230, 251, 251, 251, 1, 252, 230, 
	253, 253, 253, 1, 252, 230, 254, 254, 
	254, 1, 252, 230, 1, 255, 249, 249, 
	1, 235, 252, 230, 256, 251, 251, 1, 
	235, 252, 230, 257, 253, 253, 1, 235, 
	252, 230, 254, 254, 254, 1, 258, 1, 
	235, 259, 1, 235, 260, 1, 235, 1, 
	234, 1, 86, 87, 86, 88, 88, 88, 
	89, 90, 91, 261, 262, 261, 262, 88, 
	88, 88, 88, 88, 1, 86, 87, 86, 
	88, 88, 88, 89, 90, 91, 263, 263, 
	88, 88, 88, 88, 88, 1, 86, 87, 
	86, 88, 88, 88, 89, 90, 91, 264, 
	264, 88, 88, 88, 88, 88, 1, 86, 
	87, 86, 88, 88, 88, 89, 90, 91, 
	265, 265, 88, 88, 88, 88, 88, 1, 
	86, 87, 86, 88, 88, 88, 89, 90, 
	91, 266, 266, 88, 88, 88, 88, 88, 
	1, 86, 87, 86, 88, 88, 88, 89, 
	90, 91, 267, 267, 88, 88, 88, 88, 
	88, 1, 86, 87, 86, 88, 88, 88, 
	89, 90, 91, 268, 268, 88, 88, 88, 
	88, 88, 1, 269, 87, 269, 88, 88, 
	88, 89, 90, 270, 88, 88, 88, 88, 
	88, 1, 271, 272, 271, 3, 61, 270, 
	1, 273, 1, 274, 274, 1, 274, 274, 
	3, 61, 270, 1, 270, 275, 270, 97, 
	98, 97, 97, 277, 99, 97, 97, 97, 
	276, 278, 97, 97, 278, 97, 1, 279, 
	1, 280, 280, 1, 280, 102, 280, 97, 
	98, 97, 97, 277, 99, 97, 97, 97, 
	276, 278, 97, 97, 278, 97, 1, 281, 
	282, 281, 283, 284, 286, 287, 285, 288, 
	288, 1, 289, 1, 290, 291, 1, 292, 
	1, 293, 294, 1, 295, 1, 281, 282, 
	281, 283, 287, 296, 1, 281, 282, 281, 
	283, 287, 297, 1, 281, 282, 281, 283, 
	287, 1, 293, 298, 1, 293, 1, 290, 
	299, 1, 290, 1, 281, 282, 281, 283, 
	284, 286, 287, 300, 301, 301, 1, 281, 
	282, 281, 283, 284, 286, 287, 302, 302, 
	302, 1, 281, 282, 281, 283, 286, 287, 
	1, 304, 303, 305, 305, 1, 281, 282, 
	281, 283, 307, 287, 306, 306, 306, 1, 
	281, 282, 281, 283, 309, 287, 308, 308, 
	308, 1, 281, 282, 281, 283, 309, 287, 
	310, 310, 310, 1, 281, 282, 281, 283, 
	309, 287, 311, 311, 311, 1, 281, 282, 
	281, 283, 309, 287, 1, 312, 306, 306, 
	1, 281, 282, 281, 283, 284, 309, 287, 
	313, 308, 308, 1, 281, 282, 281, 283, 
	284, 309, 287, 314, 310, 310, 1, 281, 
	282, 281, 283, 284, 309, 287, 311, 311, 
	311, 1, 315, 1, 284, 316, 1, 284, 
	317, 1, 284, 1, 281, 282, 281, 283, 
	286, 287, 288, 288, 288, 1, 281, 282, 
	281, 283, 286, 287, 301, 301, 301, 1, 
	281, 282, 281, 283, 286, 287, 302, 302, 
	302, 1, 304, 1, 86, 87, 86, 88, 
	88, 88, 89, 90, 91, 318, 318, 88, 
	88, 88, 88, 88, 1, 86, 87, 86, 
	88, 88, 88, 89, 90, 91, 319, 319, 
	88, 88, 88, 88, 88, 1, 86, 87, 
	86, 88, 88, 88, 89, 90, 91, 320, 
	320, 88, 88, 88, 88, 88, 1, 321, 
	322, 321, 323, 324, 325, 1, 326, 327, 
	326, 3, 61, 325, 1, 328, 1, 329, 
	329, 1, 329, 329, 3, 61, 325, 1, 
	325, 330, 325, 331, 1, 332, 1, 333, 
	333, 1, 333, 333, 331, 1, 334, 335, 
	334, 336, 338, 337, 1, 86, 87, 86, 
	88, 88, 88, 89, 90, 91, 339, 339, 
	88, 88, 88, 88, 88, 1, 86, 87, 
	86, 88, 88, 88, 89, 90, 91, 340, 
	340, 88, 88, 88, 88, 88, 1, 341, 
	87, 341, 88, 88, 88, 89, 90, 342, 
	88, 88, 88, 88, 88, 1, 343, 344, 
	343, 3, 61, 342, 1, 345, 1, 346, 
	346, 1, 346, 346, 3, 61, 342, 1, 
	342, 347, 342, 97, 98, 97, 97, 99, 
	97, 97, 97, 348, 97, 97, 1, 349, 
	1, 350, 350, 1, 350, 102, 350, 97, 
	98, 97, 97, 99, 97, 97, 97, 348, 
	97, 97, 1, 351, 352, 351, 353, 355, 
	354, 1, 351, 352, 351, 353, 355, 356, 
	1, 351, 352, 351, 353, 355, 1, 54, 
	50, 50, 50, 1, 51, 52, 51, 53, 
	56, 57, 48, 50, 50, 1, 357, 50, 
	50, 1, 45, 358, 359, 48, 48, 1, 
	360, 50, 50, 1, 45, 361, 362, 48, 
	48, 1, 363, 50, 50, 1, 51, 52, 
	51, 53, 45, 49, 56, 57, 364, 48, 
	48, 1, 51, 52, 51, 53, 45, 49, 
	56, 57, 365, 48, 48, 1, 51, 52, 
	51, 53, 45, 49, 56, 57, 48, 48, 
	48, 1, 45, 361, 366, 48, 48, 1, 
	45, 361, 48, 48, 48, 1, 45, 358, 
	367, 48, 48, 1, 45, 358, 48, 48, 
	48, 1, 45, 46, 368, 48, 48, 1, 
	45, 46, 48, 48, 48, 1, 370, 369, 
	369, 369, 1, 372, 373, 371, 371, 371, 
	1, 372, 373, 374, 374, 374, 1, 372, 
	373, 375, 375, 375, 1, 372, 373, 1, 
	377, 376, 369, 369, 1, 378, 372, 373, 
	379, 371, 371, 1, 380, 1, 381, 382, 
	1, 383, 1, 384, 385, 1, 386, 1, 
	373, 387, 1, 373, 388, 1, 373, 1, 
	51, 52, 51, 53, 56, 57, 1, 384, 
	389, 1, 384, 1, 381, 390, 1, 381, 
	1, 378, 372, 373, 391, 374, 374, 1, 
	378, 372, 373, 375, 375, 375, 1, 393, 
	373, 392, 392, 392, 1, 395, 373, 394, 
	394, 394, 1, 395, 373, 396, 396, 396, 
	1, 395, 373, 397, 397, 397, 1, 395, 
	373, 1, 398, 392, 392, 1, 378, 395, 
	373, 399, 394, 394, 1, 378, 395, 373, 
	400, 396, 396, 1, 378, 395, 373, 397, 
	397, 397, 1, 401, 1, 378, 402, 1, 
	378, 403, 1, 378, 1, 377, 1, 2, 
	2, 1, 1, 0
};

static const short _tsip_machine_parser_header_Via_trans_targs[] = {
	2, 0, 3, 4, 337, 5, 8, 6, 
	7, 9, 10, 8, 13, 9, 10, 13, 
	11, 12, 14, 17, 15, 16, 18, 19, 
	17, 22, 18, 19, 22, 20, 21, 23, 
	26, 24, 25, 27, 28, 26, 27, 28, 
	31, 35, 301, 29, 30, 32, 287, 299, 
	33, 34, 35, 36, 69, 4, 285, 286, 
	40, 49, 36, 37, 40, 49, 38, 39, 
	41, 44, 42, 43, 45, 69, 4, 44, 
	49, 45, 46, 47, 48, 50, 53, 107, 
	122, 135, 208, 271, 51, 52, 54, 69, 
	53, 4, 49, 58, 54, 55, 56, 57, 
	59, 71, 65, 72, 60, 61, 62, 63, 
	64, 66, 68, 70, 67, 45, 338, 73, 
	106, 74, 77, 75, 76, 78, 93, 79, 
	91, 80, 81, 89, 82, 83, 87, 84, 
	85, 86, 88, 90, 92, 94, 102, 95, 
	98, 96, 97, 99, 100, 101, 103, 104, 
	105, 108, 109, 110, 111, 112, 113, 117, 
	113, 114, 115, 116, 118, 121, 119, 120, 
	45, 69, 121, 4, 49, 123, 124, 125, 
	126, 130, 126, 127, 128, 129, 131, 134, 
	132, 133, 45, 69, 134, 4, 49, 136, 
	137, 138, 139, 140, 144, 140, 141, 142, 
	143, 145, 148, 152, 172, 146, 147, 149, 
	158, 170, 150, 151, 152, 45, 69, 4, 
	153, 154, 49, 155, 156, 157, 159, 160, 
	168, 161, 162, 166, 163, 164, 165, 167, 
	169, 171, 173, 207, 174, 177, 187, 175, 
	176, 178, 194, 179, 192, 180, 181, 190, 
	182, 183, 188, 184, 185, 186, 189, 191, 
	193, 195, 203, 196, 199, 197, 198, 200, 
	201, 202, 204, 205, 206, 209, 258, 210, 
	211, 212, 213, 214, 215, 216, 220, 216, 
	217, 218, 219, 221, 224, 257, 254, 222, 
	223, 45, 69, 4, 225, 237, 240, 49, 
	255, 226, 227, 235, 228, 229, 233, 230, 
	231, 232, 234, 236, 238, 256, 239, 224, 
	241, 254, 242, 250, 243, 246, 244, 245, 
	247, 248, 249, 251, 252, 253, 259, 260, 
	261, 262, 69, 4, 49, 266, 262, 263, 
	264, 265, 267, 270, 268, 269, 45, 69, 
	4, 270, 49, 272, 273, 274, 278, 274, 
	275, 276, 277, 279, 282, 280, 281, 45, 
	69, 4, 283, 49, 284, 288, 289, 297, 
	290, 291, 295, 292, 293, 294, 296, 298, 
	300, 302, 336, 303, 306, 316, 304, 305, 
	307, 323, 308, 321, 309, 310, 319, 311, 
	312, 317, 313, 314, 315, 318, 320, 322, 
	324, 332, 325, 328, 326, 327, 329, 330, 
	331, 333, 334, 335
};

static const char _tsip_machine_parser_header_Via_trans_actions[] = {
	0, 0, 0, 0, 0, 0, 29, 0, 
	0, 3, 3, 0, 3, 0, 0, 0, 
	0, 0, 0, 1, 0, 0, 5, 5, 
	0, 5, 0, 0, 0, 0, 0, 0, 
	1, 0, 0, 11, 11, 0, 0, 0, 
	1, 1, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 32, 32, 32, 0, 0, 
	7, 7, 0, 0, 0, 0, 0, 0, 
	0, 1, 0, 0, 35, 35, 35, 0, 
	9, 0, 0, 0, 0, 0, 1, 1, 
	1, 1, 1, 1, 0, 0, 59, 59, 
	0, 59, 25, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 59, 27, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 59, 0, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	47, 47, 0, 47, 19, 0, 0, 0, 
	59, 0, 0, 0, 0, 0, 0, 1, 
	0, 0, 50, 50, 0, 50, 21, 0, 
	0, 0, 0, 59, 0, 0, 0, 0, 
	0, 0, 1, 1, 1, 0, 0, 0, 
	0, 0, 0, 0, 0, 41, 41, 41, 
	0, 0, 15, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 59, 0, 0, 
	0, 0, 0, 0, 1, 1, 1, 0, 
	0, 44, 44, 44, 0, 0, 0, 17, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 56, 56, 56, 23, 0, 0, 0, 
	0, 0, 0, 1, 0, 0, 62, 62, 
	62, 0, 53, 0, 0, 59, 0, 0, 
	0, 0, 0, 0, 1, 0, 0, 38, 
	38, 38, 0, 13, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0
};

static const int tsip_machine_parser_header_Via_start = 1;
static const int tsip_machine_parser_header_Via_first_final = 338;
static const int tsip_machine_parser_header_Via_error = 0;

static const int tsip_machine_parser_header_Via_en_main = 1;


/* #line 253 "./ragel/tsip_parser_header_Via.rl" */
	(void)(eof);
	(void)(tsip_machine_parser_header_Via_first_final);
	(void)(tsip_machine_parser_header_Via_error);
	(void)(tsip_machine_parser_header_Via_en_main);
	
/* #line 1122 "./src/headers/tsip_header_Via.c" */
	{
	cs = tsip_machine_parser_header_Via_start;
	}

/* #line 258 "./ragel/tsip_parser_header_Via.rl" */
	
/* #line 1129 "./src/headers/tsip_header_Via.c" */
	{
	int _klen;
	unsigned int _trans;
	const char *_acts;
	unsigned int _nacts;
	const char *_keys;

	if ( p == pe )
		goto _test_eof;
	if ( cs == 0 )
		goto _out;
_resume:
	_keys = _tsip_machine_parser_header_Via_trans_keys + _tsip_machine_parser_header_Via_key_offsets[cs];
	_trans = _tsip_machine_parser_header_Via_index_offsets[cs];

	_klen = _tsip_machine_parser_header_Via_single_lengths[cs];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( (*p) < *_mid )
				_upper = _mid - 1;
			else if ( (*p) > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _tsip_machine_parser_header_Via_range_lengths[cs];
	if ( _klen > 0 ) {
		const char *_lower = _keys;
		const char *_mid;
		const char *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( (*p) < _mid[0] )
				_upper = _mid - 2;
			else if ( (*p) > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += ((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	_trans = _tsip_machine_parser_header_Via_indicies[_trans];
	cs = _tsip_machine_parser_header_Via_trans_targs[_trans];

	if ( _tsip_machine_parser_header_Via_trans_actions[_trans] == 0 )
		goto _again;

	_acts = _tsip_machine_parser_header_Via_actions + _tsip_machine_parser_header_Via_trans_actions[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
/* #line 47 "./ragel/tsip_parser_header_Via.rl" */
	{
		tag_start = p;
	}
	break;
	case 1:
/* #line 51 "./ragel/tsip_parser_header_Via.rl" */
	{
		if(!curr_via){
			curr_via = tsip_header_Via_create_null();
		}
	}
	break;
	case 2:
/* #line 57 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_STRING(curr_via->proto_name);
	}
	break;
	case 3:
/* #line 61 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_STRING(curr_via->proto_version);
	}
	break;
	case 4:
/* #line 65 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_STRING(curr_via->host);
		if(curr_via->host && *curr_via->host == '['){
			tsk_strunquote_2(&curr_via->host, '[', ']');
		}
	}
	break;
	case 5:
/* #line 72 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_INTEGER(curr_via->port);
	}
	break;
	case 6:
/* #line 76 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_STRING(curr_via->transport);
	}
	break;
	case 7:
/* #line 80 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_INTEGER(curr_via->ttl);
	}
	break;
	case 8:
/* #line 84 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_STRING(curr_via->maddr);
	}
	break;
	case 9:
/* #line 88 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_STRING(curr_via->received);
	}
	break;
	case 10:
/* #line 92 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_STRING(curr_via->branch);
	}
	break;
	case 11:
/* #line 96 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_STRING(curr_via->comp);
	}
	break;
	case 12:
/* #line 100 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_SET_INTEGER(curr_via->rport);
	}
	break;
	case 13:
/* #line 104 "./ragel/tsip_parser_header_Via.rl" */
	{
		if(curr_via->rport <0){
			curr_via->rport = 0;
		}
	}
	break;
	case 14:
/* #line 110 "./ragel/tsip_parser_header_Via.rl" */
	{
		TSK_PARSER_ADD_PARAM(TSIP_HEADER_PARAMS(curr_via));
	}
	break;
	case 15:
/* #line 114 "./ragel/tsip_parser_header_Via.rl" */
	{
		if(curr_via){
			tsk_list_push_back_data(hdr_vias, ((void**) &curr_via));
		}
	}
	break;
	case 16:
/* #line 120 "./ragel/tsip_parser_header_Via.rl" */
	{
		
	}
	break;
/* #line 1314 "./src/headers/tsip_header_Via.c" */
		}
	}

_again:
	if ( cs == 0 )
		goto _out;
	if ( ++p != pe )
		goto _resume;
	_test_eof: {}
	_out: {}
	}

/* #line 259 "./ragel/tsip_parser_header_Via.rl" */
	
	if( cs < 
/* #line 1330 "./src/headers/tsip_header_Via.c" */
338
/* #line 260 "./ragel/tsip_parser_header_Via.rl" */
 ){
		TSK_DEBUG_ERROR("Failed to parse 'Via' header.");
		TSK_OBJECT_SAFE_FREE(curr_via);
		TSK_OBJECT_SAFE_FREE(hdr_vias);
	}
	
	return hdr_vias;
}











//========================================================
//	Via header object definition
//

static tsk_object_t* tsip_header_Via_ctor(tsk_object_t *self, va_list * app)
{
	tsip_header_Via_t *via = self;
	if(via){
		const char* proto_name = va_arg(*app, const char *);
		const char* proto_version = va_arg(*app, const char *);
		const char* transport = va_arg(*app, const char *);
		const char* host = va_arg(*app, const char *);
#if defined(__GNUC__)
		uint16_t port = (uint16_t)va_arg(*app, unsigned);
#else
		uint16_t port = va_arg(*app, uint16_t);
#endif

		if(proto_name) via->proto_name = tsk_strdup(proto_name);
		if(proto_version) via->proto_version = tsk_strdup(proto_version);
		if(transport) via->transport = tsk_strdup(transport);
		if(host) via->host = tsk_strdup(host);
		via->port = port;

		via->rport = -1;
		via->ttl = -1;
		
		TSIP_HEADER(via)->type = tsip_htype_Via;
		TSIP_HEADER(via)->serialize = tsip_header_Via_serialize;
		TSIP_HEADER(via)->get_special_param_value = tsip_header_Via_get_special_param_value;
	}
	else{
		TSK_DEBUG_ERROR("Failed to create new Via header.");
	}
	return self;
}

static tsk_object_t* tsip_header_Via_dtor(tsk_object_t *self)
{
	tsip_header_Via_t *via = self;
	if(via){
		TSK_FREE(via->branch);
		TSK_FREE(via->comp);
		TSK_FREE(via->host);
		TSK_FREE(via->maddr);
		TSK_FREE(via->proto_name);
		TSK_FREE(via->proto_version);
		TSK_FREE(via->received);
		TSK_FREE(via->sigcomp_id);
		TSK_FREE(via->transport);
		TSK_OBJECT_SAFE_FREE(TSIP_HEADER_PARAMS(via));
	}
	else{
		TSK_DEBUG_ERROR("Null Via header.");
	}

	return self;
}

static const tsk_object_def_t tsip_header_Via_def_s = 
{
	sizeof(tsip_header_Via_t),
	tsip_header_Via_ctor,
	tsip_header_Via_dtor,
	tsk_null
};
const tsk_object_def_t *tsip_header_Via_def_t = &tsip_header_Via_def_s;
